/*
LRU-2 Design Proposal for the F-CPU
Copyright (C) 2000 Michael Riepe

        This program is free software; you can redistribute it and/or modify
        it under the terms of the GNU General Public License as published by
        the Free Software Foundation; either version 2 of the License, or
        (at your option) any later version.

        This program is distributed in the hope that it will be useful,
        but WITHOUT ANY WARRANTY; without even the implied warranty of
        MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
        GNU General Public License for more details.

        You should have received a copy of the GNU General Public License
        along with this program; if not, write to the Free Software
        Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

Description:

        This LRU uses 2^n independent stages (1 per cache line) and
        a common `reference bus'.  Each stage consists of an n-bit
        register and an n-bit unsigned magnitude comparator that
        compares the number on the reference bus with the contents of
        the register.  There are two control lines, `Enable' (global) and
        `Update' (per-stage), and one output line, `Zero', indicating
        the least recently used cache line.  The register has three
        operating modes:

        - when Enable = 0 =>
          do nothing.

        - when Enable = 1 and Update = 0 =>
          when the number in the register is bigger than the number
          on the reference bus, decrement @ the next clock cycle,
          otherwise do nothing.

        - when Enable = 1 and Update = 1 =>
          put register contents on the reference bus (immediately).
          set the register to (2^n)-1 @ the next clock cycle.

        To update the LRU unit, set the global Enable line and the Update line
        for the corresponding cache line to `1', all other Update lines to `0'.

        The reference bus can be either a real bus or a n:1 multiplexer.
*/

#define DEGREE  8
#define NLINES  (1u << DEGREE)

unsigned lru[NLINES];

void
lru_init(void) {
        unsigned i;

        for (i = 0; i < NLINES; i++) {
                lru[i] = i;
        }
}

unsigned
lru_lookup(void) {
        unsigned i;

        for (i = 0; i < NLINES; i++) {
                if (lru[i] == 0) {
                        return i;
                }
        }
}

void
lru_update(unsigned line) {
        unsigned i;

        for (i = 0; i < NLINES; i++) {
                if (lru[i] > lru[line]) {
                        lru[i] -= 1;
                }
        }
        lru[line] = NLINES - 1;
}
