/*
 *
 *               THE F-CPU PROJECT
 *
 *   WHYGEE's ASSEMBLER: the syntactical analyser
 *
 *        " get ready for the headache ! "
 *
 *   Jan. 9 1999: cleaned up the GRAZER files
 *   Feb. 4 : modified some opcode fields
 *   Feb. 21: added the MOVE instruction
 *   Mar. 5 : transformed DATA into BYTE, WORD and DWORD, added ALIGN
 *            added better intra-space label resolution
 *   Mar. 8 : added OPLx and byte, word and dword constant lists
 *            with forward label references.
 *   Mar. 9 : NEW_LABEL EQU LABEL bug solved
 *   
 *   Sept. 15: adapted for F-CPU, included some basic instructions
 *   Sept. 17: fixed some bugs and added some instructions
 *
 *  (known) bugs and problems to be aware of:
 *   - Very few over/underflow checks are made.
 *     beware of limitations with 32 bit machines !!!
 *
 *  F-CPU specific codes begin around line 174
 */

%token  EQU IP LABEL LABEL_REFERENCE NEW_LABEL NUMBER ORG
%token  SPACE PROPERTIES INDEX R SIZE NOP BYTE WORD DWORD ALIGN
%token  LOOPENTRY LOOP IF MOVE MOVS MOVZ MOVI MOVIS DOTLSB BR EQUAL UNEQUAL
%token  SHIFTL SHIFTR SHIFTSR ROTR ROTL SHIFTLI SHIFTRI SHIFTSRI ROTRI
%token  ROTLI SIMD DOT_B DOT_W DOT_D DOT_Q LOGIC LOGICI ADD ADDI SUB SUBI
  /* %token xxx : add your own tokens here ! */

/* for the expression evaluator */
%left '^'
%left '|'
%left '&'
%left ">>"
%left "<<"
%left '+'
%left '-'
%left '%'
%left '/'
%left '*'
%left '~'
%left NEG

%%     /* GRAMMAR RULES */

/* TOP LEVEL */
file:  /* nothing */

/* instruction: */
| file do_before_instruction ';'
{
  unsigned char * c = &instr_tab[current_space][instr_tab_offset[current_space]];
  /* BEWARE OF THE BYTE ORDER !!! */
  add(";");
  fprintf(list_output,"%04X:%02X%02X%02X%02X, l%03d: %s\n",
     Instruction_Pointer,c0,c1,c2,c3,parse_line,instruction_description);
  *instruction_description=0; /* reset the string */
  desc_size=0;
  *c=c0;
  *(c+1)=c1;
  *(c+2)=c2;  /* write the generated instruction into the instruction buffer */
  *(c+3)=c3;
  instr_tab_offset[current_space]+=4;
  if (instr_tab_offset[current_space]>=MAX_INSTR*4)
    error ("too many instructions",NULL);
  Instruction_Pointer+=4;
}

  /* here are some "macros": */
| file LABEL
{
  referenced_label->value = Instruction_Pointer;
  referenced_label->space = current_space;
  resolve(referenced_label);
}

| file NEW_LABEL { referenced_label2=referenced_label; } EQU expr ';'
{
  referenced_label2->value = $5 ;
  referenced_label2->properties = LABEL_DECLARED ;
  referenced_label2->space = current_space;
  resolve(referenced_label2);
}

| file SIZE EQU expr ';'
{
  if ((($4 & -$4)!=$4) /* not a power of two */
     |(($4<8)|($4>32768))) /* bounds */
         error ("wrong word size",NULL);
  sizes[$2]=$4;
  fprintf(list_output,"Size #%d set to %d\n",$2,$4);
}

| file ORG expr ';' {
  Instruction_Pointer = $3;
  if (Instruction_Pointer & 3) warning ("ORG should be aligned in 4 byte boundary",NULL);
  fprintf(list_output,"Relocating the virtual instruction pointer to 0x%04X\n",
     Instruction_Pointer);
}

| file byte  bytelist ';'  { fprintf(list_output,"\n"); }
| file word  wordlist ';'  { fprintf(list_output,"\n"); }
| file dword dwordlist ';' { fprintf(list_output,"\n"); }
| file ALIGN expr ';' {
   int i,j;
   if (($3 & -$3)!=$3)
     error ("wrong align count",NULL);
   i=$3-1;
   j=~i;
   instr_tab_offset[current_space]=(instr_tab_offset[current_space]+i)& j;
   Instruction_Pointer=(Instruction_Pointer+i)& j;  /* round the pointers */
}

| file newspace ';'
;

byte:  BYTE  { fprintf(list_output,"%04X: l%03d: ",Instruction_Pointer,parse_line); }
word:  WORD  { fprintf(list_output,"%04X: l%03d: ",Instruction_Pointer,parse_line); }
dword: DWORD { fprintf(list_output,"%04X: l%03d: ",Instruction_Pointer,parse_line); }

bytelist:
  byte_item
| byte_item ',' bytelist
;

wordlist:
  word_item
| word_item ',' wordlist 
;

dwordlist:
  dword_item
| dword_item ',' dwordlist 
;

newspace:
 SPACE expr prop index {
  current_space=$2;
  if (current_space>=MAXSPACE)
    error ("space number too big",NULL);
  if (instr_tab_offset[current_space]!=0)
      error ("space already defined",NULL);
  property[current_space]=$3;
  original_index[current_space]=Instruction_Pointer = $4;
  instr_tab_offset[current_space]=0;
  fprintf(list_output,"\n  space #%d\n",current_space);
}

prop: /* nothing */ { $$=EXECUTABLEREADWRITE; }
| PROPERTIES { $$=$1; }
;

index: /* nothing */ { $$=0; }
| INDEX expr  { $$=$2; }

/* CLEANUP */
do_before_instruction:
{
  instr_tab_offset[current_space]=(instr_tab_offset[current_space]+3)& ~3; /* round the pointers */
  Instruction_Pointer=(Instruction_Pointer+3)& ~3;
  parse_line=Line_Number;     /* prevents some incoherencies*/
  c0=c1=c2=c3=0;
}
instruction
;

/* THE BIG STUFF */
instruction:

  op
| opr                    r {                                   c3|=($2<<2);          }
| oprr            r virg r {              c2|=($2<<4);         c3|=($4<<2)|($2>>4);  }
| opirr r virg imm8 virg r {              c2|=($2<<4);         c3|=($6<<2)|($2>>4);  }
| oprrr r virg r    virg r { c1|=($4<<6); c2|=($2<<4)|($4>>2); c3|=($6<<2)|($2>>4);  }
| cond condop
;

op: /* instruction(s) with no operand */
  NOP { add("nop"); c0=OP_CMOV /* ideally 0, so nothing to do */ ; }
;

opr: /* instruction with one register operand */
  LOOPENTRY { add ("loopentry "); c0=OP_LOOPENTRY; }
| movi place imm16 virg   /* warning : there's a trick here :-D */
;

oprr: /* instruction with two register operands */
  LOOP { add ("loop "); c0=OP_LOOP; }
;

opirr: 
  LOGICI   { add ("logici"); c0=OP_LOGICI; c1|=$1<<2; } size
| SHIFTLI  { add ("shiftli"); c0=OP_SHIFTLI; }   simd size
| SHIFTRI  { add ("shiftri"); c0=OP_SHIFTRI; }   simd size
| SHIFTSRI { add ("shiftsri"); c0=OP_SHIFTSRI; } simd size
| ROTRI    { add ("rotri"); c0=OP_ROTRI; }       simd size
| ROTLI    { add ("rotli"); c0=OP_ROTLI; }       simd size
| ADDI     { add ("addi"); c0=OP_ADDI; }         simd size
| SUBI     { add ("subi"); c0=OP_SUBI; }         simd size
;

oprrr:
  LOGIC   { add ("logic"); c0=OP_LOGIC; c1|=$1<<2; } size
| SHIFTL  { add ("shiftl"); c0=OP_SHIFTL; }   simd size
| SHIFTR  { add ("shiftr"); c0=OP_SHIFTR; }   simd size
| SHIFTSR { add ("shiftsr"); c0=OP_SHIFTSR; } simd size
| ROTR    { add ("rotr"); c0=OP_ROTR; }       simd size
| ROTL    { add ("rotl"); c0=OP_ROTL; }       simd size
| ADD     { add ("add"); c0=OP_ADD; }         simd size
| SUB     { add ("sub"); c0=OP_SUB; }         simd size
;


cond: /* nothing */
| if r virg                       { c1|=($2<<6); c2|=($2>>2); }
| if excl r virg                  { c1|=($2<<6); c2|=($2>>2); c1|=16; }
| if r equal expr0  virg          { c1|=($2<<6); c2|=($2>>2); }
| if r unequal expr0  virg        { c1|=($2<<6); c2|=($2>>2); c1|=16; }
| if r dotlsb equal  expr0  virg  { c1|=($2<<6); c2|=($2>>2); c1|=32; }
| if r dotlsb unequal expr0 virg  { c1|=($2<<6); c2|=($2>>2); c1|=48; }
; 

 /* instructions which can be conditional */
condop:
  MOVE {add("move");} size r virg r  { c0=OP_CMOV; c2=($4<<4); c3|=($6<<2)|($4>>4); }
| MOVS {add("movs");} size r virg r  { c0=OP_CMOV; c2=($4<<4); c3|=($6<<2)|($4>>4); c1|=4; }
| MOVZ {add("movz");} size r virg r  { c0=OP_CMOV; c2=($4<<4); c3|=($6<<2)|($4>>4); c1|=8; }
| BR {add("br ");}  r { c0=OP_BR; c3|= $3<<2;  }
;

movi:
   MOVI  { add ("movi"); c0=OP_MOVI; }
|  MOVIS { add ("movis"); c0=OP_MOVIS; }
;

simd: /* nothing */
| SIMD {add(".s"); c1|=4;}
;

excl: '!' { add("!"); } ;
r: R { $$=$1; SPRINTF("r%d",$1); };
if: IF { add("if "); };
dotlsb: DOTLSB { add(".lsb"); c1|=32; };
virg: ',' { add(", "); } ;
dot: '.' { add("."); } ;
equal: EQUAL {add("==");};
unequal: UNEQUAL {add("!="); c1|=16;};

expr0: expr { if ($1!=0) error("invalid form (should be 0)",NULL); add ("0"); };

size: dot expr {
  $$=$2;
  if ($2==sizes[0]) {/* nothing, already 0 */} else
    if ($2==sizes[1]) c1|=1; else
      if ($2==sizes[2]) c1|=2; else
        if ($2==sizes[3]) c1|=3; else
          error ("wrong register size",NULL);
      /* yeah i know it's ugly but what the heck ? */
  SPRINTF("%d ",$2);
}
| /* nothing: default to size3 */
  { c1|=3; SPRINTF(".%d ",sizes[3]);}
| DOT_B {
  if (sizes[0]==8) {}
   else { if (sizes[1]==8) c1|=1;
    else { if (sizes[2]==8) c1|=2;
     else { if (sizes[3]==8) c1|=3;
      else error ("bytes are not in the current size map",NULL); }}}
  add(".b"); 
}
| DOT_W {
  if (sizes[0]==16) {}
   else { if (sizes[1]==16) c1|=1;
    else { if (sizes[2]==16) c1|=2;
     else { if (sizes[3]==16) c1|=3;
      else error ("words are not in the current size map",NULL); }}}
  add(".w"); 
}
| DOT_D {
  if (sizes[0]==32) {}
   else { if (sizes[1]==32) c1|=1;
    else { if (sizes[2]==32) c1|=2;
     else { if (sizes[3]==32) c1|=3;
      else error ("double words are not in the current size map",NULL); }}}
  add(".d"); 
}
| DOT_Q {
  if (sizes[0]==64) {}
   else { if (sizes[1]==64) c1|=1;
    else { if (sizes[2]==64) c1|=2;
     else { if (sizes[3]==64) c1|=3;
      else error ("quad words are not in the current size map",NULL); }}}
  add(".q"); 
}
;

place: '[' expr ']' {
  int i=$2;
  if ((i<0)|(i>8)) error("wrong register place",NULL);
  SPRINTF("[%d] ",i);
  c1|=i<<6;
  c0|=i>>2;
}
;

imm8: expr
{
  if (($1>127)|($1<= -128))
   error ("imm8 overflow",NULL);
  c1|=($1>>4) & 15;
  c2|=($1<<4);
  SPRINTF("0x%02X",$1);
}
| NEW_LABEL
{
  /* this code adds a node to the relocation list */
  register struct undefined_node *p = mem_increase(sizeof(*p));
  p->next=referenced_label->undefined_list;
  p->where=instr_tab_offset[current_space]; 
  p->reloc_type=RELOC_imm8;
  p->space=current_space;
  referenced_label->undefined_list=p;
  add("{undefined}");
}
;

imm16: expr
{
  if (($1>32767)|($1<= -32768))
   error ("imm16 overflow",NULL);
  c1|=($1<<2);
  c2|=($1>>6);
  c3|=($1>>14) & 3;
  SPRINTF("0x%04x",$1);
}
| NEW_LABEL
{
  register struct undefined_node *p = mem_increase(sizeof(*p));
  p->next=referenced_label->undefined_list;
  p->where=instr_tab_offset[current_space];   /* before it is incremented */
  p->reloc_type=RELOC_imm16;
  p->space=current_space;
  referenced_label->undefined_list=p;
  add("{undefined}");
}
;

byte_item: expr
{
  if (($1>127)|($1<= -128))
    error ("byte overflow",NULL);
  c0= $1;
  fprintf(list_output,"0x%02X ",c0);
  /* write the data to the buffer */
  instr_tab[current_space][instr_tab_offset[current_space]++]=c0;
  if (instr_tab_offset[current_space]>=MAX_INSTR*4)
    error ("too many instructions",NULL);
  Instruction_Pointer++;
}
| NEW_LABEL
{
  register struct undefined_node *p = mem_increase(sizeof(*p));
  p->next=referenced_label->undefined_list;
  p->where=instr_tab_offset[current_space];   /* before it is incremented */
  p->reloc_type=RELOC_byte;
  p->space=current_space;
  referenced_label->undefined_list=p;
  fprintf(list_output,"{?} ");
  instr_tab_offset[current_space]++;
  Instruction_Pointer++;
}
;

word_item: expr
{
  if (($1>32767)|($1<= -32768))
    error ("word overflow",NULL);
  c0=$1;
  c1=$1>>8;
  fprintf(list_output,"0x%02X%02X ",c1,c0);
  /* write the data to the buffer */
  instr_tab[current_space][instr_tab_offset[current_space]++]=c0;
  instr_tab[current_space][instr_tab_offset[current_space]++]=c1;
  if (instr_tab_offset[current_space]>=MAX_INSTR*4)
    error ("too many instructions",NULL);
  Instruction_Pointer+=2;
}
| NEW_LABEL
{
  register struct undefined_node *p = mem_increase(sizeof(*p));
  p->next=referenced_label->undefined_list;
  p->where=instr_tab_offset[current_space];   /* before it is incremented */
  p->reloc_type=RELOC_word;
  p->space=current_space;
  referenced_label->undefined_list=p;
  fprintf(list_output,"{ud} ");
  instr_tab_offset[current_space]+=2;
  Instruction_Pointer+=2;
}
;

dword_item: expr
{
  c0= $1;
  c1= $1>>8;
  c2= $1>>16;
  c3= $1>>24;
  fprintf(list_output,"%02X%02X%02X%02X ",c3,c2,c1,c0);
  /* write the data to the buffer */
  instr_tab[current_space][instr_tab_offset[current_space]  ]=c0;
  instr_tab[current_space][instr_tab_offset[current_space]+1]=c1;
  instr_tab[current_space][instr_tab_offset[current_space]+2]=c2;
  instr_tab[current_space][instr_tab_offset[current_space]+3]=c3;
  instr_tab_offset[current_space]+=4;
  if (instr_tab_offset[current_space]>=MAX_INSTR*4)
    error ("too many instructions",NULL);
  Instruction_Pointer+=4;
}
| NEW_LABEL
{
  register struct undefined_node *p = mem_increase(sizeof(*p));
  p->next=referenced_label->undefined_list;
  p->where=instr_tab_offset[current_space];   /* before it is incremented */
  p->reloc_type=RELOC_dword;
  p->space=current_space;
  referenced_label->undefined_list=p;
  fprintf(list_output,"{undefd} ");
  instr_tab_offset[current_space]+=4;
  Instruction_Pointer+=4;
}
;

/* expression evaluator with arithmetic and boolean operations */
/* BEWARE: NO UNDER/OVERFLOW CHECK IS PERFORMED !!! */
expr: NUMBER  { $$ = yy_number;  }
    | LABEL_REFERENCE { $$ = referenced_label->value; }
    | IP { $$ = Instruction_Pointer; }
    | expr '^' expr  { $$ = $1 ^ $3; }
    | expr '|' expr  { $$ = $1 | $3; }
    | expr '&' expr  { $$ = $1 & $3; }
    | expr ">>" expr { $$ = $1 >> $3; }
    | expr "<<" expr { $$ = $1 << $3; }
    | expr '+' expr  { $$ = $1 + $3; }
    | expr '-' expr  { $$ = $1 - $3; }
    | expr '%' expr  { $$ = $1 % $3; }
    | expr '/' expr  { $$ = $1 / $3; }
    | expr '*' expr  { $$ = $1 * $3; }
    | '~' expr  { $$ = ~ $2; }
    | '-' expr %prec NEG { $$ = - $2; }
    | '(' expr ')'   { $$ = $2; }
    ;

%%
