 /*

MEMORY.C

Whygee's GNL memory manager (C) Yann Guidon 29 mars 2000
This  is  free software; see the GPL for copying condi
tions.  There is NO warranty; not even for MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.

whygee@f-cpu.org

Note: because of the nature of the algorithms used here,
there is no need of a FREE function. we have our own
memory reuse function so there are few memory leaks.

The strategy is simple : the linked lists are not mixed.
the elements of each list are stored in separately malloc()ed
blocks. One can mix several lists if they are flushed together.

 */

/* for malloc etc */
#include <stdlib.h>
#include <stdio.h>

#define MALLOC_GRANULARITY 4096
/* we treat memory with this granularity from the OS point of view */

/* so we can run on a 64-bit machine : */
#define POINTER_ALIGN (sizeof(char *))
#define POINTER_ALIGN_MASK (POINTER_ALIGN-1)

/*
 structure of one block :
 word 0 : pointer to the next block, zero if last.
 int : number of bytes used in the block, granularity of POINTER_ALIGN
 */

typedef struct struct_block_header{
  void * next;
  int size; /* number of used bytes */
}block_header ;

#define HEADSIZE ((sizeof(block_header)+POINTER_ALIGN_MASK)& ~POINTER_ALIGN_MASK)

/* important static data : */
block_header *ll_garbage=NULL;
/* each linked list should have a pointer like this :
void *ll_root_block;
*/

int block_count=0;

void * new_block(void) {
  block_header *ptr;
  if (ll_garbage==NULL) {
    /* ol'good malloc : */
    ptr=calloc(1,MALLOC_GRANULARITY);
    if (ptr==NULL)
      error(-2,"\nerror : can't allocate memory\n");
    ptr->size=HEADSIZE;
    block_count++;
    if (!(block_count & 7))
      fprintf(stderr," * %d KB allocated\n",(block_count*MALLOC_GRANULARITY)>>10);
    return ptr;
  }  /* else */
  /* use a spare block from the garbage linked list : */
  ptr=ll_garbage;
  ll_garbage=ll_garbage->next;
  return ptr;
}

void * my_alloc(block_header **ll_root, int alloc_size) {
  char *ptr2;
  block_header *ptr = *ll_root;

  /* 1) round up */
  alloc_size=(alloc_size+POINTER_ALIGN)& ~POINTER_ALIGN_MASK;

  /* 2) little reality check */
  if (alloc_size>MALLOC_GRANULARITY-HEADSIZE)
    error(-3,"\nerror : block too big\n");

  if (*ll_root==NULL) {
    ptr=new_block();
    *ll_root=ptr;
    ptr->next=NULL;
    ptr->size+=alloc_size;
    return ((char*)ptr)+HEADSIZE;
  }

  /* 3) test if the current block has enough room */
  while ((ptr->size+alloc_size)>=MALLOC_GRANULARITY) {
    if (ptr->next==NULL) {
      ptr->next=new_block();
      ptr=ptr->next;
      ptr->size+=alloc_size;
      return ((char*)ptr)+HEADSIZE;
    }
    /* else */
    ptr=ptr->next;
  }

  /* return the good pointer */
  ptr2=(char *)ptr;
  ptr2+=ptr->size;
  ptr->size+=alloc_size;
  return ptr2;
}


/* free a linked list of blocks */
void ll_free(block_header **ll_root) {
  block_header *p=*ll_root;
  if (p!=NULL) {
loop_llfree:
    memset(((char*)p)+HEADSIZE,0,p->size);
    p->size=HEADSIZE;
    if (p->next!=NULL) {
      p=p->next;
      goto loop_llfree;
    }
    p->next=ll_garbage;
    ll_garbage=*ll_root;
    *ll_root=NULL;
  }
}
