#include "stdlib.h"
#include "fcpudefs.h"

//
//
//

void fcpu_utl_free( void** p )
{
    if( *p )
        free( *p );

    *p = NULL;
}

void* fcpu_utl_malloc( UL64 size )
{
    if( size > 1024L * 1024L * 1024L )
        return NULL;

    return malloc( (unsigned long)size );
}

void fcpu_utl_memset( void* p, char d, UL64 s )
{
    unsigned char*  puc;

    puc = (unsigned char*)p;
    while( s > 0 )
    {
        *puc = d;

        puc++;
        s--;        
    }
}

//
//
//

char fcpu_utl_upper_case( char ch )
{
    if( ( ch >= 'a' ) && ( ch <= 'z' ) )
    {
        // ASSUME 'a' > 'A'
        return (char)( ( ch - 'a' ) + 'A' );
    }

    return ch;
}

UL64 fcpu_utl_length_of_string( char* psz )
{
    UL64 i;

    i = 0;

    while( *psz )
    {
        psz++;
        i++;
    }

    return i;
}

UL64 fcpu_utl_length_of_string_buffer( char* psz )
{
    return fcpu_utl_length_of_string( psz ) + 1;
}

void fcpu_utl_clear_string( char* psz, UL64 m )
{
    while( m )
    {
        *psz = 0;

        psz++;
        m--;
    }
}

char fcpu_utl_is_empty_string( char* psz )
{
    if( 0 == *psz )
        return 1L;

    return 0L;
}

void fcpu_utl_append_char_to_string( char ch, char* psz, UL64 m )
{
    UL64 i;

    if( m == 0 )
        return;

    if( m == 1 )
    {
        *psz = 0;
        return;
    }   

    i = 0;
    while( *psz != 0 )
    {
        psz++;
        i++;
    }

    if( i < ( m - 1 ) )
    {
        *( psz + 0 ) = ch;
        *( psz + 1 ) = 0;
    }
}

void fcpu_utl_copy_string( char* pszd, UL64 md, char* pszs )
{
    UL64 i;

    if( md == 0 )
        return;

    if( md == 1 )
    {
        *pszd = 0;
        return;
    }

    i = 0;
    while( ( *pszs != 0 ) && ( i < ( md - 1 ) ) )
    {
        *pszd = *pszs;

        pszd++;
        pszs++;
        i++;
    }
    
    *pszd = 0;
}

char fcpu_utl_check_string_inside_string_at_pos_i( char* psz0, char* psz1, UL64 pos )
{
    UL64 l0;
    UL64 l1;    

    l0 = fcpu_utl_length_of_string( psz0 );
    l1 = fcpu_utl_length_of_string( psz1 );

    if( l1 > l0 )
        return 0L;

    if( pos >= l0 )
        return 0L;

    psz0 += pos;

    while( ( *psz0 != 0 ) && ( *psz1 != 0 ) )
    {
        if( fcpu_utl_upper_case( *psz0 ) != fcpu_utl_upper_case( *psz1 ) )
            return 0L;

        psz0++;
        psz1++;
    }

    if( *psz1 == 0 )
        return 1L;

    return 0L;
}

char fcpu_utl_check_string_inside_string_i( char* psz0, char* psz1 )
{
    UL64 i;

    for( i = 0; i < fcpu_utl_length_of_string( psz0 ); i++ )
    {
        if( fcpu_utl_check_string_inside_string_at_pos_i( psz0, psz1, i ) )
            return 1L;
    }

    return 0L;
}

char fcpu_utl_compare_strings_i( char* psz0, char* psz1 )
{
    while( ( *psz0 != 0 ) && ( *psz1 != 0 ) )
    {
        if( fcpu_utl_upper_case( *psz0 ) != fcpu_utl_upper_case( *psz1 ) )
            return 0L;

        psz0++;
        psz1++;
    }

    if( ( *psz0 == 0 ) && ( *psz1 == 0 ) )
        return 1L;

    return 0L;
}

//
//
//

UL64 fcpu_utl_convert_digit_to_constant( char digit, UL64 base, char* perr )
{
    UL64 x;

    if( perr )
        *perr = 0L;

    if( ( digit >= '0' ) && ( digit <= '9' ) )
    {
        x = (UL64)( digit - '0' );
    }
    else if( ( digit >= 'A' ) && ( digit <= 'Z' ) )
    {
        x = (UL64)( digit - 'A' + 10 );
    }
    else if( ( digit >= 'a' ) && ( digit <= 'z' ) )
    {
        x = (UL64)( digit - 'a' + 10 );
    }
    else
    {
        if( perr )
            *perr = 1L;

        return 0L;
    }

    if( x >= base )
    {
        if( perr )
            *perr = 1L;

        return 0L;
    }

    return x;
}

UL64 fcpu_utl_convert_integer_string_to_constant( char* psz, char* perr )
{
    UL64    x;
    UL64    c;
    UL64    base;
    char    neg;
    char    err;

    base    = 10;
    neg     = 0;

    if( perr )
        *perr = 0;

    if( *psz == 0 )
    {
        if( perr )
            *perr = 1L;

        return 0L;
    }
    else if( *( psz + 1 ) != 0 )
    {
        if( ( *( psz + 0 ) == '0' ) && ( *( psz + 1 ) == 'x' ) )
        {
            base = 16;
            psz += 2;
        }
    }
    
    if( *psz == '$' )
    {
        base = 16;
        psz++;
    }

    if( *psz == '%' )
    {
        base = 2;
        psz++;
    }

    if( *psz == '+' )
    {
        base = 10;
        psz++;
    }

    if( *psz == '-' )
    {
        base = 10;
        neg  = 1;
        psz++;
    }

    err = 0L;
    x   = 0L;

    while( ( *psz ) && ( !err ) )
    {
        x = base * x;

        c = fcpu_utl_convert_digit_to_constant( *psz, base, &err );
        x = x + c;

        psz++;
    }

    if( err )
    {
        if( perr )
            *perr = 1L;

        return 0L;
    }

    if( neg )
        x = ( ~x ) + 1;

    return x;
}

char fcpu_utl_check_string_is_float_constant( char* psz )
{
    if( fcpu_utl_check_string_inside_string_i( psz, "." ) )
        return 1L;

    return 0L;
}

UL64 fcpu_utl_convert_float_string_to_constant( char* psz, char* perr )
{
    double  fd;
    float   fs;
    UL64    fi;
    UL64    l;
    UL64    i;
    char*   psz0;
    char    sizech;

    l = fcpu_utl_length_of_string( psz );
    if( l < 4 )
    {
        if( perr )
            *perr = 1L;
        
        return 0L;
    }

    sizech = *( psz + l - 1 );
    if( ( sizech != 'f' ) && ( sizech != 'd' ) )
    {
        if( perr )
            *perr = 1L;

        return 0L;
    }

    if( !fcpu_utl_check_string_inside_string_i( psz, "." ) )
    {
        if( perr )
            *perr = 1L;

        return 0L;
    }

    psz0 = (char*)fcpu_utl_malloc( l );
    if( psz0 )
    {        
        for( i = 0; i < ( l - 1 ); i++ )
            *( psz0 + i ) = *( psz + i );
        *( psz0 + l - 1 ) = 0;

        fd = atof( psz0 );

        fcpu_utl_free( (void**)&psz0 );
    }
    else
        fd = 0.0;

    fs = (float)fd;

    if( sizech == 'f' )
        fi = (UL64)( *( (UL32*)&fs ) );
    else
        fi = (UL64)( *( (UL64*)&fd ) );

    return fi;
}

UL64 fcpu_utl_convert_string_to_constant( char* psz, char* perr )
{
    UL64 x;
    char err;

    x = fcpu_utl_convert_float_string_to_constant( psz, &err );
    if( err )
    {
        x = fcpu_utl_convert_integer_string_to_constant( psz, &err );
        if( err )
            x = 0;
    }
    
    if( perr )
    {
        if( err )
            *perr = 1;
        else
            *perr = 0;
    }

    return x;
}

//
//
//

void* fcpu_utl_get_next_element_of_linked_list( void* p )
{   
    return *(void**)p;
}

void fcpu_utl_append_to_linked_list( void** pproot, void* pnew )
{   
    void* p;

    if( !*pproot )
    {
        *pproot = pnew;
    }
    else
    {
        p = *pproot;
        while( p )
        {
            if( NULL == fcpu_utl_get_next_element_of_linked_list( p ) )
            {
                *(void**)p = pnew;
                break;
            }

            p = fcpu_utl_get_next_element_of_linked_list( p );
        }
    }
}


