#ifndef __FCPUDEFS_H_
#define __FCPUDEFS_H_

#define UL64    unsigned __int64
#define I64     __int64
#define UL32    unsigned long
#define I32     long
#define UL08    unsigned char

#ifndef FALSE
#define FALSE   0L
#endif

#ifndef TRUE
#define TRUE    1L
#endif

#ifndef NULL
#define NULL    0L
#endif

//
//
//

#define BIT( x ) ( ( (UL64)1 ) << ( x ) )

//
//
//

#define FCPU_OP_UNKNOWN      0xFF

/* not a real "operation" in the sense that the data are not modified */
#define FCPU_OP_MOVE         0
/* so NOP is detected with an instruction equal to zero */

#define FCPU_OP_NOP 0
/* followed by 3 empty bytes, but the 6 last bits (dest reg) can be tested now to simplify things. */

/* Integer arithmetics */
#define FCPU_OP_ADD          2
#define FCPU_OP_SUB          4
#define FCPU_OP_MUL          6
#define FCPU_OP_DIV          8
#define FCPU_OP_MOD          10
#define FCPU_OP_ADDSUB       12
#define FCPU_OP_MAC          13

/* not really arithmetic... */
#define FCPU_OP_POPCOUNT     14

/* INC-based instructions */
#define FCPU_OP_CMPL         24
#define FCPU_OP_CMPLE        26
#define FCPU_OP_MAX          28
#define FCPU_OP_MIN          30
#define FCPU_OP_SORT         21

/* LNS operations */
#define FCPU_OP_LADD         32
#define FCPU_OP_LSUB         33

/* SHL ("shuffler") operations */
#define FCPU_OP_SHIFTL       36
#define FCPU_OP_SHIFTR       38
#define FCPU_OP_SHIFTRA      40
#define FCPU_OP_ROTL         42
#define FCPU_OP_ROTR         44
#define FCPU_OP_BITOP        46
#define FCPU_OP_BITREV       48
/* SIMD and byte-shuffling */
#define FCPU_OP_MIX          52
#define FCPU_OP_EXPAND       53
#define FCPU_OP_SDUP         54
#define FCPU_OP_PERMUTE      126

/* ROP2 unit (not complete : combine is new) */
#define FCPU_OP_LOGIC        56
#define FCPU_OP_COMBINE_OR   58
#define FCPU_OP_COMBINE_AND  59

/* FP (not complete) */
#define FCPU_OP_FADD         64
#define FCPU_OP_FSUB         65
#define FCPU_OP_FMUL         66
#define FCPU_OP_FDIV         71
#define FCPU_OP_FMAC         75
#define FCPU_OP_FADDSUB      76

/* LSU, dome forms ar also in intermediary format */
#define FCPU_OP_LOAD         80
#define FCPU_OP_LOADF        82
#define FCPU_OP_STORE        84
#define FCPU_OP_STOREF       86
#define FCPU_OP_CACHEMM      88

#define FCPU_OP_SCATTER      127
#define FCPU_OP_GATHER       128

/* MISC : */
/* SRB : */
#define FCPU_OP_LOADM        108
#define FCPU_OP_STOREM       110

/* control-related */
#define FCPU_OP_JMPA         116


/*
 intermediary format : RR
 ------------------------
 can belong to either RRR or I8RR (RRR seems better)
 count=15
*/

/* INC-based instructions */
#define FCPU_OP_INC          16
/*
   the following functions can be encoded in the bits 12 and 13 !
   #define FCPU_OP_DEC
   #define FCPU_OP_NEG
   #define FCPU_OP_ABS
*/


/* hole here. */

#define FCPU_OP_SCAN         20

/* LNS operations */
#define FCPU_OP_L2INT        34
#define FCPU_OP_INT2L        35

/* SIMD and byte-shuffling */
#define FCPU_OP_BYTEREV      50

/* FP, not complete */
#define FCPU_OP_F2INT        67
#define FCPU_OP_INT2F        68
#define FCPU_OP_FIAPRX       69
#define FCPU_OP_FSQRTIAPRX   70
#define FCPU_OP_FSQRT        72
#define FCPU_OP_FLOG         73
#define FCPU_OP_FEXP         74

/* MISC : */
/* SPR : */
#define FCPU_OP_GET          104
#define FCPU_OP_PUT          106

/* control-related */
#define FCPU_OP_LOOP         117    /* this isn't going to be RR only in some time... */

/*
 Format one : I8RR
 -----------------
 (plus sign bit in a 9th bit)
 count=27

 challenge : map it so it corresponds to the RRR format whenever possible,
so the difference is "only" one bit in most cases -> easier decoding.

*/

/* Integer arithmetics */

#define FCPU_OP_ADDI         3
#define FCPU_OP_SUBI         5
#define FCPU_OP_MULI         7
#define FCPU_OP_DIVI         9
#define FCPU_OP_MODI         11

/* INC-based instructions */

#define FCPU_OP_CMPLI        25
#define FCPU_OP_CMPLEI       27
#define FCPU_OP_MAXI         29
#define FCPU_OP_MINI         31

/* not really arithmetic */
#define FCPU_OP_POPCOUNTI    15

/* SHL ("shuffler") operations */
#define FCPU_OP_SHIFTLI      37
#define FCPU_OP_SHIFTRI      39
#define FCPU_OP_SHIFTRAI     41
#define FCPU_OP_ROTLI        43
#define FCPU_OP_ROTRI        45
#define FCPU_OP_BITOPI       47
#define FCPU_OP_BITREVI      49
/* SIMD and byte-shuffling */
#define FCPU_OP_SDUPI        55

/* ROP2 unit (not complete : combine is new) */
#define FCPU_OP_LOGICI       57
#define FCPU_OP_COMBINE_ORI  60
#define FCPU_OP_COMBINE_ANDI 61

/* LSU */
#define FCPU_OP_LOADI        81
#define FCPU_OP_LOADIF       83
#define FCPU_OP_STOREI       85
#define FCPU_OP_STOREIF      87
#define FCPU_OP_CACHEMMI     88

/* SRB : */
#define FCPU_OP_LOADMI       109
#define FCPU_OP_STOREMI      111

/*
 Format two : I16R
 -----------------
 count=12
 */

/* MISC : */
/* close to FCPU_OP_MOVE */
#define FCPU_OP_LOADCONS     96   /* 4-opcode range so we can create 256-bit immediates */
#define FCPU_OP_LOADCONSX    100  /* idem */
/* total : 8 opcodes */

/* SPR : */
#define FCPU_OP_GETI         105
#define FCPU_OP_PUTI         107

/* control-related */
#define FCPU_OP_LOADADDR     114
#define FCPU_OP_LOADADDRI    115

/*
 Format three : I24
 ------------------
 count=10
*/

/* SRB : */
#define FCPU_OP_SRB_SAVE     112
#define FCPU_OP_SRB_RESTORE  113
/* control-related */
#define FCPU_OP_SYSCALL      118
#define FCPU_OP_RFE          119
#define FCPU_OP_HALT         120
#define FCPU_OP_SERIALIZE    121

#define FCPU_OP_VLIW         122
#define FCPU_OP_VLIW0        122   /* should be rounded on a 2-bit boundary ! */
#define FCPU_OP_VLIW1        123   /* ultimately, would be opcode #252-255 */
#define FCPU_OP_VLIW2        124
#define FCPU_OP_VLIW3        125

// !!! Opcode map differs from manual in th INC-based ops
// !!! As long as emulator is not changed
// !!! these opcodes are defined here
#define FCPU_OP_ABS          200
#define FCPU_OP_DEC          201
#define FCPU_OP_NEG          202
#define FCPU_MAX_REGS               64
#define FCPU_MAX_OPCODES            256
#define FCPU_SR_LAST_SR             45
#define FCPU_MAXSIZE                8

#define FCPU_LOG_NUM_TLB_ENTRIES	5
#define FCPU_NUM_TLB_ENTRIES		( 1 << FCPU_LOG_NUM_TLB_ENTRIES )

#define FCPU_NUM_VMID_BITS          16
#define FCPU_LOG_NUM_VMIDC_ENTRIES  2
#define FCPU_NUM_VMIDC_ENTRIES      ( 1 << FCPU_LOG_NUM_VMIDC_ENTRIES )

#define FCPU_EXCEPTION_RESET                BIT( 0 )
#define FCPU_EXCEPTION_HALT                 BIT( 1 )
#define FCPU_EXCEPTION_INVALID_INSTRUCTION  BIT( 2 )
#define FCPU_EXCEPTION_PAGEFAULT            BIT( 3 )
#define FCPU_EXCEPTION_PROTECTIONFAULT      BIT( 4 )
#define FCPU_EXCEPTION_ALIGNMENTFAULT       BIT( 5 )
#define FCPU_EXCEPTION_DIVISION_BY_ZERO     BIT( 6 )
#define FCPU_EXCEPTION_TIMER_COUNT_ZERO     BIT( 7 )

// Exception Vectors
//
// The FCPU jumps to ( content of( FCPU_SR_TRAP_BASE ) || ( vector << 6 ) )
// on fault/exception/interrupt
//
#define FCPU_TRAP_VECTOR_INVALID_INSTRUCTION    0
#define FCPU_TRAP_VECTOR_PROTECTIONFAULT        1
#define FCPU_TRAP_VECTOR_ALIGNMENTFAULT         2
#define FCPU_TRAP_VECTOR_DIVISION_BY_ZERO       3
#define FCPU_TRAP_VECTOR_TIMER_COUNT_ZERO       4

// FCPU Special Registers
//
// The following registers are protected. This means, that FCPU executes
// an FCPU_EXCEPTION_PROTECTIONFAULT if the are written if msr.rwsr = 0.
//
// There are also a few bits inside of FCPU_SR_MSR which are protected.
// The FCPU executes FCPU_EXCEPTION_PROTECTIONFAULT if the user mode application
// tries to change them.
//

/* number of SRs that are implemented in this model */
#define FCPU_SR_NUMBERS             0
#define FCPU_SR_NUMBERS_VAL         FCPU_SR_LAST_SR  /* hardwired */
#define FCPU_SR_FAMILY              1
#define FCPU_SR_FAMILY_VAL          0x454D554C41544F52 /* "EMULATOR" (big endian, hardwired) */
#define FCPU_SR_STEPPING            2
#define FCPU_SR_STEPPING_VAL        1  /* hardwired */
#define FCPU_SR_MAX_SIZE            3
#define FCPU_SR_MAX_SIZE_VAL        FCPU_MAXSIZE  /* hardwired */
#define FCPU_SR_SIZE_1              4
#define FCPU_SR_SIZE_1_VAL          1 /* R/W */
#define FCPU_SR_SIZE_2              5
#define FCPU_SR_SIZE_2_VAL          2 /* R/W */
#define FCPU_SR_SIZE_3              6
#define FCPU_SR_SIZE_3_VAL          4 /* R/W */
#define FCPU_SR_SIZE_4              7
#define FCPU_SR_SIZE_4_VAL          8 /* R/W */
#define FCPU_SR_MAX_CHUNK_SIZE      8
#define FCPU_SR_MAX_CHUNK_SIZE_VAL  8 /* hardwired */
#define FCPU_SR_CYCLE               9 /* R/W, Value is dynamic, incremented every cycle. */
#define FCPU_SR_PAGING              10 /* Protected, R/W, Controls the paged memory. */
/* IRQ, TRAP and SYSCALL jump tables : all are R/W in protected mode - only. */
#define FCPU_SR_IRQ_BASE            11 /* For the hardware interrupt requests */
#define FCPU_SR_IRQ_SIZE            12
#define FCPU_SR_TRAP_BASE           13 /* faults and system errors */
#define FCPU_SR_TRAP_SIZE           14
#define FCPU_SR_SYSCALL_BASE        15 /* OS system call */
#define FCPU_SR_SYSCALL_SIZE        16
#define FCPU_SR_TLBMISS_BASE        17 /* TLB miss trap */

/* The 8 URL registers must be modified to point to the manual, sources, patches etc.
   64 characters max. for a 64-bit version, 32 chars for a 32-bit version...
   These registers are hardwired, format is ASCII and padded with 0s. */
#define FCPU_SR_URL           18
#define FCPU_SR_URL_SIZE      8
#define FCPU_SR_URL_VAL       "http://www.f-cpu.de/emu/" /* or wherever your version is located */

/* these ones are freshly added : */
#define FCPU_SR_TIME_SLICE_COUNTER  26
#define FCPU_SR_TIME_SLICE_LIMIT    27
#define FCPU_SR_CMB_BASE            28
#define FCPU_SR_EVENT_COUNTER_0_CFG 29 
#define FCPU_SR_EVENT_COUNTER_0     30
#define FCPU_SR_EVENT_COUNTER_1_CFG 31
#define FCPU_SR_EVENT_COUNTER_1     32
#define FCPU_SR_EVENT_COUNTER_2_CFG 33
#define FCPU_SR_EVENT_COUNTER_2     34
#define FCPU_SR_EVENT_COUNTER_3_CFG 35
#define FCPU_SR_EVENT_COUNTER_3     36
#define FCPU_SR_EVENT_COUNTER_4_CFG 37
#define FCPU_SR_EVENT_COUNTER_4     38
#define FCPU_SR_EVENT_COUNTER_5_CFG 39
#define FCPU_SR_EVENT_COUNTER_5     40
#define FCPU_SR_EVENT_COUNTER_6_CFG 41
#define FCPU_SR_EVENT_COUNTER_6     42
#define FCPU_SR_EVENT_COUNTER_7_CFG 43
#define FCPU_SR_EVENT_COUNTER_7     44

#define FCPU_SR_FLAGS               45

#define FCPU_SR_TLB_BASE            0x8000000000000000
#define FCPU_TLB_ENTRY_SIZE         0x10

//
//
//

#define FCPU_MEM_MEMORY_SIZE   ( 16L * 1024L * 1024L )

//
//
//
#define FCPU_FLAGS_VMID             0x000000000000FFFF
#define FCPU_FLAGS_KERNELMODE       0x0000000000010000
#define FCPU_FLAGS_INTERRUPTENABLE  0x0000000000020000

#define FCPU_TLB_ACCESS_MODE_R      0
#define FCPU_TLB_ACCESS_MODE_W      1
#define FCPU_TLB_ACCESS_MODE_RW     2
#define FCPU_TLB_ACCESS_MODE_X      3

//
//
//
typedef void (*PXF)( void* );
typedef void (*PSF)( void*, UL64, UL64, UL64*, UL64* );
typedef UL64 (*PSEF)( UL64 );
typedef UL64 (*PGRF)( void*, UL64, UL64 );
typedef void (*PPRF)( void*, UL64, UL64, UL64 );

// VM-ID Cache Entry
typedef struct _VMIDCE {
    UL64    vmid:FCPU_NUM_VMID_BITS;
    UL64    lru:FCPU_LOG_NUM_VMIDC_ENTRIES;
    UL64    valid:1;
} VMIDCE, *PVMIDCE;

// TLB Entry
typedef struct _TLBE {
    UL64 log_addr;
    UL64 phy_addr;
    UL64 page_size;                     /* size of page in bytes = 2 ^ page_size = 1 << page_size */
    UL64 hit_counter;                   /* incremented whenever the address matches. */
    UL64 valid:1;                       /* tlb entry contains valid adress translation data */
    UL64 access_rights:2;               /* r=0, w=1, rw=2, x=3*/
    UL64 cachable:1;                    /* if 0, don't seek in the cache */
    UL64 lru:FCPU_LOG_NUM_TLB_ENTRIES;
    UL64 vmid_handle:2;
} TLBE, *PTLBE;

// CMB
typedef struct _CMB {
    UL64    instr_ptr;
    UL64    reg[ FCPU_MAX_REGS ];
    UL64    cycle_counter;          /* counts all cycles this task executed */
    UL64    time_slice_limit;       /* value to be loaded into time_slice_counter on task switch */
    UL64    time_slice_counter;     /* is loaded with time_slice_limit on task switch. */
                                    /* counting from 1->0 causes FCPU_EXCEPTION_TIMER_COUNT_ZERO exception */
    UL64    next_cmb_ptr;           /* pointer to next CMB in list */
    UL64    sr_syscall_base;
    UL64    sr_syscall_limit;
} CMB, *PCMB;

// FCPU Registers
typedef struct _FCPU {
    UL64    exception;
    UL64    instr_ptr;  
    UL64    instr_reg;
    VMIDCE  vmidc[ FCPU_NUM_VMIDC_ENTRIES ];
    TLBE    tlb[ FCPU_NUM_TLB_ENTRIES ];
    UL64    reg[ FCPU_MAX_REGS ];
    UL64    sr_cycle;
    UL64    sr_paging;
    UL64    sr_irq_base;
    UL64    sr_irq_size;
    UL64    sr_trap_base;
    UL64    sr_trap_size;
    UL64    sr_syscall_base;
    UL64    sr_syscall_size;
    UL64    sr_tlb_miss_base;
    UL64    sr_time_slice_counter;
    UL64    sr_time_slice_limit;
    UL64    sr_cmb_base;
    UL64    sr_event_counter_0_cfg;
    UL64    sr_event_counter_0;
    UL64    sr_event_counter_1_cfg;
    UL64    sr_event_counter_1;
    UL64    sr_event_counter_2_cfg;
    UL64    sr_event_counter_2;
    UL64    sr_event_counter_3_cfg;
    UL64    sr_event_counter_3;
    UL64    sr_event_counter_4_cfg;
    UL64    sr_event_counter_4;
    UL64    sr_event_counter_5_cfg;
    UL64    sr_event_counter_5;
    UL64    sr_event_counter_6_cfg;
    UL64    sr_event_counter_6;
    UL64    sr_event_counter_7_cfg;
    UL64    sr_event_counter_7;
    UL64    sr_flags;
    // Emulator specific 
    UL08    opcode;
    UL08    sizeflg;
    UL08    r3;
    UL08    r2;
    UL08    r1;
    UL64    imm16;
    UL64    imm8;
    UL64    imm6;
    //
    UL64    sb[ 4 ];
    UL64    iss[ 4 ];
    UL64    fss[ 4 ];
    UL64    mask[ 4 ];
    UL64    msbmask[ 4 ];
    PSEF    psef[ 4 ];
    PGRF    pigrf[ 4 ];
    PPRF    piprf[ 4 ];
    PGRF    pfgrf[ 4 ];
    PPRF    pfprf[ 4 ];
    PXF     pxf[ FCPU_MAX_OPCODES ];
    PSF     psf[ FCPU_MAX_OPCODES ][ 4 ];
} FCPU, *PFCPU;

//
// FCPUASM exports
//

char fcpu_asm_assemble_file( char* );

//
// FCPUCPU exports
//

void fcpu_cpu_reset( PFCPU );

void fcpu_cpu_clock( PFCPU );

//
// FCPUMEM exports
//

UL64 fcpu_mem_get_n_bytes_le( UL64 a, UL64 n );

UL64 fcpu_mem_get_n_bytes_be( UL64 a, UL64 n );

void fcpu_mem_put_n_bytes_le( UL64 a, UL64 n, UL64 d );

void fcpu_mem_put_n_bytes_be( UL64 a, UL64 n, UL64 d );

//
// FCPUUTL exports
// 

void fcpu_utl_free( void** p );

void* fcpu_utl_malloc( UL64 size );

void fcpu_utl_memset( void* p, char d, UL64 s );

char fcpu_utl_upper_case( char ch );

UL64 fcpu_utl_length_of_string( char* psz );

UL64 fcpu_utl_length_of_string_buffer( char* psz );

void fcpu_utl_clear_string( char* psz, UL64 m );

char fcpu_utl_is_empty_string( char* psz );

void fcpu_utl_append_char_to_string( char ch, char* psz, UL64 m );

void fcpu_utl_copy_string( char* pszd, UL64 md, char* pszs );

char fcpu_utl_check_string_inside_string_at_pos_i( char* psz0, char* psz1, UL64 pos );

char fcpu_utl_check_string_inside_string_i( char* psz0, char* psz1 );

char fcpu_utl_compare_strings_i( char* psz0, char* psz1 );

UL64 fcpu_utl_convert_digit_to_constant( char digit, UL64 base, char* perr );

UL64 fcpu_utl_convert_string_to_constant( char* psz, char* perr );

void* fcpu_utl_get_next_element_of_linked_list( void* p );

void fcpu_utl_append_to_linked_list( void** pproot, void* pnew );

#endif // __FCPUDEFS_H_
