-- iatest5.vhdl - Testbench for F-CPU IAdd Unit
-- Copyright (C) 2000 Michael Riepe <michael@s...>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

-- $Id: iatest5.vhdl,v 1.3 2000/11/10 03:20:20 michael Exp $

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_textio.all;
use IEEE.numeric_std.all;
use std.textio.all;

entity IAdd_test is
end IAdd_test;

architecture Arch_1 of IAdd_test is
        constant WIDTH : natural := 64;

        signal M : std_ulogic_vector(4 downto 0) := (others => '0');
        signal A, B, H, L : std_ulogic_vector(WIDTH-1 downto 0) := (others => '0');
        signal Yl, Y8l, Yh, Y8h : std_ulogic_vector(WIDTH-1 downto 0) := (others => '0');

        procedure writestr (s : string) is
                variable lout : line;
        begin
                write(lout, s);
                writeline(output, lout);
        end writestr;

        procedure do_report (lbl : string;
                                                 x, y : std_ulogic_vector) is
                variable lout : line;
        begin
                write(lout, string'("WHOA THERE!!!")); writeline(output, lout);
                write(lout, string'("A := ")); write(lout, A); writeline(output, lout);
                write(lout, string'("B := ")); write(lout, B); writeline(output, lout);
                write(lout, string'("M := ")); write(lout, M); writeline(output, lout);
                write(lout, string'("H := ")); write(lout, H); writeline(output, lout);
                write(lout, string'("L := ")); write(lout, L); writeline(output, lout);
                write(lout, lbl);
                write(lout, string'(" := "));
                write(lout, x);
                writeline(output, lout);
                write(lout, lbl);
                write(lout, string'(" /= "));
                write(lout, y);
                writeline(output, lout);
        end do_report;

        procedure check_numeric (lbl : string;
                                                         x : std_ulogic_vector;
                                                         y : natural) is
                variable tmp : std_ulogic_vector(x'range);
                variable lout : line;
        begin
                tmp := std_ulogic_vector(to_unsigned(y mod 2**x'length, x'length));
                if x /= tmp then
                        do_report(lbl, x, tmp);
                end if;
        end check_numeric;

        procedure check_logic (lbl : string;
                                                   a, b : std_ulogic_vector) is
                alias x : std_ulogic_vector(a'length downto 1) is a;
                alias y : std_ulogic_vector(b'length downto 1) is b;
                variable lout : line;
        begin
                assert a'length = b'length;
                for i in x'range loop
                        next when y(i) = '-';
                        next when x(i) = y(i);
                        do_report(lbl, x, y);
                        return;
                end loop;
        end check_logic;
begin
        -- module under test
        mut : entity work.IAdd
                generic map (WIDTH => WIDTH)
                port map (
                        A => A, B => B, Sub => M(0), Sat => M(1),
                        U08 => M(2), U16 => M(3), U32 => M(4),
                        Y8l => Y8l, Y8h => Y8h, Yl => Yl, Yh => Yh
                );

        -- choose correct output
        L <= Yl when M(2) = '1' else Y8l;
        H <= Yh when M(2) = '1' else Y8h;

        -- driver process
        process
                variable av, bv, tmp : std_ulogic_vector(WIDTH-1 downto 0);
                variable left, right, bits, carry, simd, res : natural;
                variable lout : line;
        begin
                M <= "UUU00";   -- add mode
                writestr("*** testing add instruction ***");
                for gran in 3 downto 0 loop
                        simd := 8 * 2**gran;
                        write(lout, string'("*** testing "));
                        write(lout, simd);
                        write(lout, string'("-bit mode ***"));
                        writeline(output, lout);
                        -- set simd mode bits
                        for n in 0 to 2 loop
                                if gran > n then
                                        M(n + 2) <= '1';
                                else
                                        M(n + 2) <= '0';
                                end if;
                        end loop;
                        -- main test loop
                        av := (others => 'X');
                        bv := (others => 'X');
                        for slice in 0 to WIDTH/4-1 loop
                                -- calculate slice range and width
                                right := 4 * slice; left := right + 4;
                                if left mod simd = 0 then
                                        -- leftmost 4-bit slice, strip MSBit
                                        left := left - 1;
                                end if;
                                bits := left - right + 1;
                                -- location of carry-out bit
                                carry := right - right mod simd;
                                -- carry-in on/off loop
                                for k in 0 to 1 loop
                                        -- turn carry-in on/off
                                        if right /= 0 then
                                                if k = 0 then
                                                        av(right-1) := '0';
                                                        bv(right-1) := '0';
                                                else
                                                        av(right-1) := '1';
                                                        bv(right-1) := '1';
                                                end if;
                                        end if;
                                        -- input values loops
                                        for i in 0 to 2**bits-1 loop
                                                av(left downto right) := std_ulogic_vector(to_unsigned(i, bits));
                                                A <= av;
                                                for j in 0 to 2**bits-1 loop
                                                        bv(left downto right) := std_ulogic_vector(to_unsigned(j, bits));
                                                        B <= bv;
                                                        wait for 1 ns;
                                                        -- test normal output
                                                        res := i + j;
                                                        if right mod simd /= 0then
                                                                res := res + k;
                                                        end if;
                                                        check_numeric("L", L(left downto right), res);
                                                        if gran = 0 then
                                                                check_numeric("Y", Yl(left downto right), res);
                                                        end if;
                                                        -- test carry output
                                                        tmp := (others => '0');
                                                        for n in 0 to WIDTH/simd-1 loop
                                                                tmp(simd * n) := '-';
                                                        end loop;
                                                        if left mod simd = simd - 1 then
                                                                -- this is theleftmost slice
                                                                if res >= 2**bits then
                                                                        tmp(carry) := '1';
                                                                else
                                                                        tmp(carry) := '0';
                                                                end if;
                                                        end if;
                                                        check_logic("H", H, tmp);
                                                        if gran = 0 then
                                                                check_logic("Y", Yh, tmp);
                                                        end if;
                                                end loop;
                                        end loop;
                                end loop;
                                -- clear slice
                                av(left downto right) := (others => 'X');
                                bv(left downto right) := (others => 'X');
                                if right /= 0 then
                                        av(right-1) := 'X';
                                        bv(right-1) := 'X';
                                end if;
                        end loop;
                        -- test saturation
                        write(lout, string'("*** testing "));
                        write(lout, simd);
                        write(lout, string'("-bit saturation ***"));
                        writeline(output, lout);
                        M(1) <= '1';
                        for i in WIDTH/simd-1 downto 0 loop
                                right := simd * i;
                                left := right + simd - 1;
                                av := (others => 'X');
                                bv := (others => 'X');
                                for j in left downto right loop
                                        av(j) := '0';
                                        bv(j) := '0';
                                        A <= av; B <= bv; wait for 1 ns;
                                        tmp := (others => 'X');
                                        tmp(left downto j + 1) := (others => '1');
                                        if j = right then tmp(j) := '0'; end if;
                                        check_logic("L", Yl, tmp);
                                        av(j) := '1';
                                        bv(j) := '1';
                                        A <= av; B <= bv; wait for 1 ns;
                                        tmp(left downto right) := (others => '1');
                                        check_logic("L", Yl, tmp);
                                        bv(j) := '0';
                                end loop;
                        end loop;
                        M(1) <= '0';
                end loop;

                M <= "UUU01";   -- sub mode
                writestr("*** testing sub instruction ***");
                for gran in 3 downto 0 loop
                        simd := 8 * 2**gran;
                        write(lout, string'("*** testing "));
                        write(lout, simd);
                        write(lout, string'("-bit mode ***"));
                        writeline(output, lout);
                        -- set simd mode bits
                        for n in 0 to 2 loop
                                if gran > n then
                                        M(n + 2) <= '1';
                                else
                                        M(n + 2) <= '0';
                                end if;
                        end loop;
                        -- main test loop
                        av := (others => 'X');
                        bv := (others => 'X');
                        for slice in 0 to WIDTH/4-1 loop
                                -- calculate slice range and width
                                right := 4 * slice; left := right + 4;
                                if left mod simd = 0 then
                                        -- leftmost 4-bit slice, strip MSBit
                                        left := left - 1;
                                end if;
                                bits := left - right + 1;
                                -- location of carry-out bit
                                carry := right - right mod simd;
                                -- carry-in on/off loop
                                for k in 0 to 1 loop
                                        -- turn carry-in on/off
                                        if right /= 0 then
                                                if k = 0 then
                                                        av(right-1) := '1';
                                                        bv(right-1) := '0';
                                                else
                                                        av(right-1) := '0';
                                                        bv(right-1) := '1';
                                                end if;
                                        end if;
                                        -- input values loops
                                        for i in 0 to 2**bits-1 loop
                                                av(left downto right) := std_ulogic_vector(to_unsigned(i, bits));
                                                A <= av;
                                                for j in 0 to 2**bits-1 loop
                                                        bv(left downto right) := std_ulogic_vector(to_unsigned(j, bits));
                                                        B <= bv;
                                                        wait for 1 ns;
                                                        -- test normal output
                                                        res := i - j + 2**bits;
                                                        if right mod simd /= 0then
                                                                res := res - k;
                                                        end if;
                                                        check_numeric("L", L(left downto right), res);
                                                        if gran = 0 then
                                                                check_numeric("Y", Yl(left downto right), res);
                                                        end if;
                                                        -- test carry output
                                                        tmp := (others => '0');
                                                        for n in 0 to WIDTH/simd-1 loop
                                                                tmp(simd * n) := '-';
                                                        end loop;
                                                        if left mod simd = simd - 1 then
                                                                -- this is theleftmost slice
                                                                if res < 2**bits then
                                                                        tmp(carry) := '1';
                                                                else
                                                                        tmp(carry) := '0';
                                                                end if;
                                                        end if;
                                                        check_logic("H", H, tmp);
                                                        if gran = 0 then
                                                                check_logic("Y", Yh, tmp);
                                                        end if;
                                                end loop;
                                        end loop;
                                end loop;
                                -- clear slice
                                av(left downto right) := (others => 'X');
                                bv(left downto right) := (others => 'X');
                                if right /= 0 then
                                        av(right-1) := 'X';
                                        bv(right-1) := 'X';
                                end if;
                        end loop;
                        -- test saturation
                        write(lout, string'("*** testing "));
                        write(lout, simd);
                        write(lout, string'("-bit saturation ***"));
                        writeline(output, lout);
                        M(1) <= '1';
                        for i in WIDTH/simd-1 downto 0 loop
                                right := simd * i;
                                left := right + simd - 1;
                                av := (others => 'X');
                                bv := (others => 'X');
                                for j in left downto right loop
                                        av(j) := '1';
                                        bv(j) := '0';
                                        A <= av; B <= bv; wait for 1 ns;
                                        tmp := (others => 'X');
                                        tmp(left downto j + 1) := (others => '0');
                                        if j = right then tmp(j) := '1'; end if;
                                        check_logic("L", Yl, tmp);
                                        av(j) := '0';
                                        bv(j) := '1';
                                        A <= av; B <= bv; wait for 1 ns;
                                        tmp(left downto right) := (others => '0');
                                        check_logic("L", Yl, tmp);
                                        bv(j) := '0';
                                end loop;
                        end loop;
                        M(1) <= '0';
                end loop;

                -- stop simulation
                writestr("*** simulation complete ***");
                wait;
        end process;
end Arch_1;
