-- simd_cla.vhdl -- SIMD-enabled Carry Look-Ahead Unit
-- Copyright (C) 2000 Michael Riepe <michael@s...>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

-- $Id: simd_cla.vhdl,v 1.2 2000/11/03 15:58:36 michael Exp $

library IEEE;
use IEEE.std_logic_1164.all;

entity SIMD_CLA is
        port (
                -- propagate/generate inputs
                Gi, Pi : in std_ulogic_vector(3 downto 0) := (others => '0');
                -- SIMD mode
                U1, U2 : in std_ulogic := '0';
                -- carry out
                Co : out std_ulogic_vector(3 downto 0)
        );
end SIMD_CLA;

architecture Struct_1 of SIMD_CLA is
        component AND2 is -- assume d=1
                port (A, B : in std_ulogic; Y : out std_ulogic);
        end component;

        component AND3 is -- assume d=1
                port (A, B, C : in std_ulogic; Y : out std_ulogic);
        end component;

        component AND4 is -- assume d=1
                port (A, B, C, D : in std_ulogic; Y : out std_ulogic);
        end component;

        component OR2 is -- assume d=1
                port (A, B : in std_ulogic; Y : out std_ulogic);
        end component;

        component OR3 is -- assume d=1
                port (A, B, C : in std_ulogic; Y : out std_ulogic);
        end component;

        signal t1, t2, t3, t4, t5 : std_ulogic;
begin
        -- Behavior:
        --      Co <= (
        --              0 => '0',
        --              1 => (U1 and Gi(0)),
        --              2 => (U2 and Gi(1))
        --                or (U2 and Pi(1) and Gi(0)),
        --              3 => (U1 and Gi(2))
        --                or (U2 and Pi(2) and Gi(1))
        --                or (U2 and Pi(2) and Pi(1) and Gi(0))
        --      );
        co_0 : Co(0) <= '0';
        co_1 : AND2 port map (U1, Gi(0), Co(1));
        tmp1 : AND2 port map (U2, Gi(1), t1);
        tmp2 : AND3 port map (U2, Pi(1), Gi(0), t2);
        co_2 : OR2  port map (t1, t2, Co(2));
        tmp3 : AND2 port map (U1, Gi(2), t3);
        tmp4 : AND3 port map (U2, Pi(2), Gi(1), t4);
        tmp5 : AND4 port map (U2, Pi(2), Pi(1), Gi(0), t5);
        co_3 : OR3  port map (t3, t4, t5, Co(3));
end Struct_1;

-- vi: set ts=4 sw=4 : please
