-- shl_in.vhdl -- Bit Shuffling Unit, Input Stage
-- Copyright (C) 2001 Michael Riepe <michael@stud.uni-hannover.de>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

-- $Id$

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.numeric_std.all;

entity Shl_Input is
	generic (
		WIDTH : natural := 8	-- do not change
	);
	port (
		-- operand to shift
		A : in std_ulogic_vector(WIDTH-1 downto 0);
		-- shift count
		B : in std_ulogic_vector(2 downto 0);
		-- shift mode
		Left : in std_ulogic := '0';	-- shift/rotate left
		Right : in std_ulogic := '0';	-- shift/rotate right
		Extend : in std_ulogic := '0';	-- set together with Right!
		Normal : in std_ulogic := '0';	-- no operation (for byterev)
		Reverse : in std_ulogic := '0';	-- bit reverse
	--
		-- double-width output
		Y : out std_ulogic_vector(2*WIDTH-1 downto 0)
	);
begin
	assert WIDTH = 8
		report "WIDTH must be 8"
		severity failure;
end Shl_Input;

architecture Behave_1 of Shl_Input is
	function v_or (a : in std_ulogic_vector) return std_ulogic is
		variable res : std_ulogic;
	begin
		res := '0';
		for i in a'range loop
			case to_X01(a(i)) is
				when '1' => return '1';
				when 'X' => res := 'X';
				when '0' => null;
			end case;
		end loop;
		return res;
	end v_or;
begin
	process (A, B, Left, Right, Extend, Normal, Reverse)
		variable t : std_ulogic_vector(WIDTH-1 downto 0);
		variable sbits : std_ulogic_vector(7 downto 0);
		variable k : integer;
		variable i_left, i_right, i_extend, i_reverse, i_normal : std_ulogic_vector(2*WIDTH-1 downto 0);
	begin
		-- 3-to-8 decoder for second operand (may be separated)
		sbits := (
			0 => not B(2) and not B(1) and not B(0),
			1 => not B(2) and not B(1) and     B(0),
			2 => not B(2) and     B(1) and not B(0),
			3 => not B(2) and     B(1) and     B(0),
			4 =>     B(2) and not B(1) and not B(0),
			5 =>     B(2) and not B(1) and     B(0),
			6 =>     B(2) and     B(1) and not B(0),
			7 =>     B(2) and     B(1) and     B(0)
		);

		-- i_normal (identity operation for byterev)
		i_normal := (others => '0');
		for i in WIDTH-1 downto 0 loop
			i_normal(i) := A(i) and Normal;
		end loop;

		-- i_left (left shift/rotate)
		i_left := (others => '0');
		for i in 2*WIDTH-1 downto 0 loop
			t := (others => '0');
			for j in WIDTH-1 downto 0 loop
				k := i - j;
				if (k >= 0) and (k < WIDTH) then
					t(j) := A(j) and sbits(k) and Left;
				end if;
			end loop;
			i_left(i) := v_or(t);
		end loop;

		-- i_right (right shift/rotate)
		i_right := (others => '0');
		for i in 2*WIDTH-1 downto 0 loop
			t := (others => '0');
			for j in WIDTH-1 downto 0 loop
				k := j - i + 8;
				if (k >= 0) and (k < WIDTH) then
					t(j) := A(j) and sbits(k) and Right;
				end if;
			end loop;
			i_right(i) := v_or(t);
		end loop;

		-- i_extend (sign extension for arithmetic right shift)
		i_extend := (others => '0');
		for i in 15 downto 8 loop
			i_extend(i) := A(7) and v_or(sbits(7 downto 15-i)) and Extend;
		end loop;

		-- i_reverse (bitrev)
		--
		-- NOTE! This differs from the F-CPU manual (v0.2)!
		-- instead of `bit_reverse(a) >> (size - b)', it calculates
		-- `bit_reverse(a) >> (size - b - 1)'.	That way, the range
		-- for the `b' operand becomes [0...size-1] instead of
		-- [1...size], just like in regular shift/rotate operations.
		-- It's easier (and faster!) to decode, too.
		--
		i_reverse := (others => '0');
		for i in 2*WIDTH-1 downto 0 loop
			t := (others => '0');
			for j in WIDTH-1 downto 0 loop
				k := i + j - 8;
				if (k >= 0) and (k < WIDTH) then
					t(j) := A(j) and sbits(k) and Reverse;
				end if;
			end loop;
			i_reverse(i) := v_or(t);
		end loop;

		-- output
		Y <= i_left or i_right or i_extend or i_reverse or i_normal;
	end process;
end Behave_1;

-- vi: set ts=4 sw=4 equalprg="fmt -72 -p--": please
