#include <string.h>
#include <stdio.h>

#include "emu.h"
#include "lsu.h"
enum { false, true } ;

/* MACRO Defitions
 * ***************
 */ 

#define LSU_MAX_LRU 0xFF

/* type Definitions
 * ****************
 *
 * lsu_line_t : a line of the data cache 
 */ 
  
typedef struct lsu_line_s {
  U64 addr ;              /* in some cases, we really need it ! */
  unsigned char reg_num ;   /* register the line is associated with */
  unsigned char valid ; /* is this line valid */
  unsigned char dirty ; /* has this line been modified since it has been fetched */
#ifdef LEAST_RECENTLY_USED
  unsigned char lru_count ; /* age since the line hasn't been accessed */
#endif
  unsigned char data[LSU_LINEWIDTH] ;   /* the data itself */
} lsu_line_t ;

/* module-wide-variable definitions
 * ********************************
 *  line[LSU_NUMLINE] : holds the data-cache lines
 *  reg_to_line_map  : table for the reverse mapping
 *  the_line         : pointer to the current line
 *  L1address        : current address (is it usefull?)
 *  pointer_register : ID of the register we are using to access the memory 
 */

lsu_line_t line[LSU_NUMLINE] ;

unsigned char reg_to_line_map[FCPU_NUM_REG] ;

lsu_line_t * lsu_curr_line ;

typedef enum { idle = 0, fetching, writing } state_t ;

state_t state ;

BOOL waitDataCache = false ;

lsu_line_t * lsu_get_new_line() ;
BOOL lsu_fetch_data(lsu_line_t *f_line) ;
BOOL lsu_write_back(lsu_line_t *w_line) ;

/* the function-pointers : communications with the outside world 
 * *********************
 *
 */
/*
lsu_notify_wait_t lsu_notify_wait ;
lsu_interface_read L1_data_read ;
lsu_interface_write L1_data_write ;
lsu_register_mapping core_reg_map ;
*/

void lsu_doPrefetch() {
  
   switch(state) {
   case fetching:
     if(lsu_fetch_data(lsu_curr_line)) {
       lsu_curr_line = lsu_get_new_line() ;  /* now we can go on to the next line */
       state = (lsu_curr_line->dirty ? writing : idle) ; /* if next line is dirty, then get ready to write it back */
     }
     break ;   
   default:     //idle
     if(!lsu_curr_line->dirty)
       break;   // ok we really have nothing to do
     state = writing;
   case writing:
     if (lsu_write_back(lsu_curr_line)) {
       state = idle ; 
     }
     break ;
    }
}

/* function
 *    lsu_fetch_data :
 *        called to fetch a line
 *
 *  returns true when the line has been fetched
 * 
 */

BOOL lsu_fetch_data(lsu_line_t *f_line) {
  U64 * addr = &(f_line->addr) ;

  if (L1DataGetData(*addr, &(f_line->data[*addr & LSU_LO_ADDR_MASK]))) {
    f_line->valid = 1 ;
    f_line->dirty = 0 ;
    return true ;
  }
  return false ;

}

/* function
 *    lsu_write_back :
 *        called when writing back a line (state == writing)
 *
 *  returns true when the line has been written
 */

BOOL lsu_write_back(lsu_line_t *w_line) {
  U64 * addr = &(w_line->addr) ;
  
  if(L1DataDoWriteMem(*addr, &(w_line->data[*addr & LSU_LO_ADDR_MASK]))) {
    w_line->dirty = 0 ;
    return true ;
  }
  return false ;
}

void lsu_init( /*lsu_notify_wait_t the_notify_function,
	     lsu_interface_read the_read_function, 
	     lsu_interface_write the_write_function,
	     lsu_register_mapping the_regmap_function */) 
{
  memset(reg_to_line_map, 0xFF, FCPU_NUM_REG) ;
  lsu_curr_line = line ;
  line[0].reg_num = 0xFF ;
  line[0].valid = 0 ;
  line[0].dirty = 0 ;
#ifdef LEAST_RECENTLY_USED
  line[0].lru_count = 0 ;
#endif
  memcpy(line+1, line, (LSU_NUMLINE-1)*sizeof(lsu_line_t)) ;
  /*  lsu_notify_wait = the_notify_function ? the_notify_function : lsu_null_notify ;
  L1_data_read = the_read_function ; 
  L1_data_write = the_write_function ; 
  core_reg_map = the_regmap_function ; */
}

#ifdef LEAST_RECENTLY_USED
void lsu_update_lru() {
  int i = 0 ;
  do {
    if ( line[i].lru_count++ == LSU_MAX_LRU ) line[i].lru_count = LSU_MAX_LRU ;
  } while (++i<LSU_NUMLINE) ;
}
#endif

#ifdef LEAST_RECENTLY_USED
lsu_line_t * lsu_get_new_line() {     /* get the next line to fetch - LRU version */
  int imax = 0, max = line[0].lru_count ;
  int i = 1 ;

  while (i<LSU_NUMLINE) {
    if (line[i].lru_count > max) {
      imax = i ; 
      max = line[i].lru_count ;
    }
    ++i ;
  }
  return imax ;
}
#else

lsu_line_t * lsu_get_new_line() {
  int line_index = lsu_curr_line - line ;
  return  line+(line_index+1)%LSU_NUMLINE ;
}

#endif


BOOL lsu_search_line(U64 addr, int *index)
{
  int i;
  for(i = 0; i < LSU_NUMLINE; i++)
    if(line[i].addr == addr)
      {
	*index = i;
	return true;
      }
  return false;
}
unsigned char lsu_do_read(U64 addr, int num_reg) {

  unsigned int index = reg_to_line_map[num_reg] ;

/* lsu_update_lru() ; */
  if (index >= LSU_NUMLINE   && !lsu_search_line(addr & LSU_HI_ADDR_MASK, &index) ) 
    {
      /* ######  MISS !  ###### */
      index = lsu_curr_line - line ;
      waitDataCache = true ; /* the core has to wait for us */
      reg_to_line_map[lsu_curr_line->reg_num] = 0xFF ; /* expulse the old register from this line */
      if (state == idle) { /* we have to wait for the fetch/write unit to be idle */
	lsu_curr_line->valid = 0 ; /* invalidate the line */
	reg_to_line_map[num_reg] = index ; /* the line belongs now to the register (direct mapping) */
	lsu_curr_line->reg_num = num_reg ;  /*  "   "      "     "   "  "     "     (reverse mapping) */
	lsu_curr_line->addr = addr & LSU_HI_ADDR_MASK ;
	state = fetching ; /* requests the fetch/write thread to fetch our line */
      } 
      return 0 ; /* must wait next cycle to return the data */ 
    }
  /* ######  HIT !  ###### */
  /* line[index]->lru_count = 0 ; */
  if (! line[index].valid) {  /* our line is still being fetched from memory */
    // assert(line+index == lsu_curr_line) ;
    waitDataCache = true ;
    return 0 ;
  }
  waitDataCache = false ;
  return line[index].data[addr & LSU_LO_ADDR_MASK] ;
}

void lsu_do_write(U64 addr, int num_reg, unsigned char data) {

  unsigned int index = reg_to_line_map[num_reg] ;
 /* lsu_update_lru() ; */

  if (index >= LSU_NUMLINE) { 
    /* ###### MISS ! ###### */
    index = lsu_curr_line - line ;
    waitDataCache = true ;
    reg_to_line_map[lsu_curr_line->reg_num] = 0xFF ; /* expulse the old register */
    if (state == idle) {   /* we have to wait for the fetch/write unit to be idle */ 
      lsu_curr_line->valid = 0 ; /* invalidate this line  */
      reg_to_line_map[num_reg] = index ; /* this line now belongs to the register (direct mapping) */
      lsu_curr_line->reg_num = num_reg ;  /* this line now belongs to the register (reverse mapping) */
      lsu_curr_line->addr = addr ;
      state = fetching ;
    }
    waitDataCache = true ;
  }    
  else if (! line[index].valid) {
    // assert(line+index == lsu_curr_line) ;
    waitDataCache = true ;
  }
  else {
    waitDataCache = false ; 
    /* line->lru_count = 0 ; */
    line[index].dirty = 1 ;
    line[index].data[addr & LSU_LO_ADDR_MASK] = data ;
  }
}

/* 
 *   Special operations:
 *   lsu_break_reg_line_mapping : break the association between a register and a line
 *   lsu_loadaddr : force the attribution of a line to a register and start fetching
 *                 except when the current is being written back
 */

void lsu_break_reg_line_mapping(int reg_num) {
  
  lsu_line_t * the_line ;
  unsigned int index = reg_to_line_map[reg_num] ;
  if (index < LSU_NUMLINE) {   /* the register is associated with a line */
    the_line = line + index ; 
    the_line->reg_num = 0xFF ;
    reg_to_line_map[reg_num] = 0xFF ;
  } 
}

void lsu_loadaddr(int reg_num, U64 addr) {
  unsigned int index = reg_to_line_map[reg_num] ;
  if (index < LSU_NUMLINE) {
    if (state == idle ) {
      index = lsu_curr_line - line ;
      lsu_curr_line->valid = 0 ; /* invalidate this line  */
      reg_to_line_map[reg_num] = index ; /* this line now belongs to the register (direct mapping) */
      lsu_curr_line->reg_num = reg_num ;  /* this line now belongs to the register (reverse mapping) */
      lsu_curr_line->addr = addr ;
      state = fetching ; 
    }
  }
}

void lsu_flush()
{
  int i;
  for(i = 0; i < LSU_NUMLINE; i++)
	while(!lsu_write_back(&line[i])){ wait();};
  
}
void lsu_null_notify(int ncycle, char * reason) { 

}

void lsu_default_notify(int ncycle, char * reason) {

  printf("Waited %d cycles, reason : %s\n", ncycle, reason) ;

}

void lsu_dump() {
  int i = 0 ;
  int j ;

    printf("--------------       ---------------------------------------------------\n"
	   "! Reg ! line !       ! line ! Reg !      Addr      ! V ! D ! Data \n"
	   "--------------       ---------------------------------------------------\n") ;
  while((i < FCPU_NUM_REG)&&(i < LSU_NUMLINE)) {
    if(i < FCPU_NUM_REG ) 
      printf("! %02d  !  %02hhX  !    ", i , reg_to_line_map[i]) ;
    else 
      printf("                      ") ;
    if(i < LSU_NUMLINE) {
      printf("%s!  %02d  ! %02hhX  !%016llX! %1hhX !  %1hhX ! ", 
	     i==lsu_curr_line-line?"-->":"   ", i, line[i].reg_num, line[i].addr, line[i].valid, line[i].dirty) ;
      
      for( j = 0 ; j < LSU_LINEWIDTH; j++) 
	printf("%02hhX", line[i].data[j] ) ;
    }
    putchar('\n') ;
    ++i ;
  }
}
