/*
 * fcpu_decode.c -- F-CPU Single-Instruction Disassembler
 * Copyright (C) 2001 - 2003 Michael Riepe <michael@stud.uni-hannover.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

static const char rcsid[] = "@(#) $Id: fcpu_decode.c,v 1.21 2003/02/03 21:52:18 michael Exp $";

#if HAVE_CONFIG_H
#include <config.h>
#endif

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#include <stdarg.h>
#include <stdio.h>
#include <assert.h>

#include <fcpu_opcodes/fcpu_opcodes.h>

#define DECODE_LONG_NAMES		0
#define DECODE_SIZE_NUMERIC		0
#define DECODE_SCAN_CANONICAL	0
#define DECODE_IMM_FORMAT		"0x%x"	/* "%u" */
#define DECODE_DEFCS_INT		""		/* ".o" */
#define DECODE_DEFCS_FP			""		/* ".d" */

#define MAX_INSN_LEN 16

int fcpu_used_register[3];
int fcpu_used_register_nb;

typedef struct table table_t;
struct table {
	const char		*name;
	int				(*decode)(const table_t*, char *buf, size_t len, unsigned op);
	unsigned char	args;
	unsigned char	simd;
};

const unsigned arg_mask[ARG_last] = {
	[ARG_NONE] = 0x00000000,
	[ARG_R]    = 0x0000003f,
	[ARG_RO]   = 0x00000fff,
	[ARG_RR]   = 0x00000fff,
	[ARG_U16R] = 0x003fffff,
	[ARG_S16R] = 0x003fffff,
	[ARG_S17R] = 0x007fffff,
	[ARG_ORR]  = 0x0003ffff,
	[ARG_RRO]  = 0x0003ffff,
	[ARG_RRR]  = 0x0003ffff,
	[ARG_U8RR] = 0x000fffff,
	[ARG_S8RR] = 0x000fffff,
	[ARG_S9RR] = 0x001fffff,
};

static int
decode_reg(char *buf, size_t len, unsigned n) {
	sprintf(buf, "r%u", n & 0x3f);
	fcpu_used_register[fcpu_used_register_nb++] = n & 0x3f;
	return strlen(buf);
}

static int
decode_imm(char *buf, size_t len, int val, int bits) {
	size_t i = 0;
	unsigned mask;

	assert(bits);
	buf[i++] = '$';
	if (bits < 0) {
		mask = 1u << -bits;
		val &= mask - 1;
		val ^= mask >> 1;
		val -= mask >> 1;
		if (val < 0) {
			buf[i++] = '-';
			val = -val;
		}
	}
	else {
		mask = 1u << bits;
		val &= mask - 1;
	}
	sprintf(buf + i, DECODE_IMM_FORMAT, (unsigned)val);
	return strlen(buf);
}

static int
decode_args(char *buf, size_t len, unsigned args, unsigned op) {
    size_t i = 0;

    switch (args) {
		case ARG_NONE:	/* (halt, rfe, srb_save, srb_restore, serialize, nop) */
			break;
		case ARG_R:		/* reg (loopentry) */
			i += decode_reg(buf + i, len - i, op >> 0);
			break;
		case ARG_RO:	/* reg[,reg] (jmp) */
			i += decode_reg(buf + i, len - i, op >> 6);
			if (op & 0x3f) {
				buf[i++] = ',';
				buf[i++] = ' ';
				i += decode_reg(buf + i, len - i, op >> 0);
			}
			break;
		case ARG_RR:	/* reg,reg (common) */
			i += decode_reg(buf + i, len - i, op >> 6);
			buf[i++] = ',';
			buf[i++] = ' ';
			i += decode_reg(buf + i, len - i, op >> 0);
			break;
		case ARG_U16R:	/* uimm16,reg (loadcons.n, geti, puti, syscall, trap) */
			i += decode_imm(buf + i, len - i, op >> 6, 16);
			buf[i++] = ',';
			buf[i++] = ' ';
			i += decode_reg(buf + i, len - i, op >> 0);
			break;
		case ARG_S16R:	/* simm16,reg (loadconsx.n) */
			i += decode_imm(buf + i, len - i, op >> 6, -16);
			buf[i++] = ',';
			buf[i++] = ' ';
			i += decode_reg(buf + i, len - i, op >> 0);
			break;
		case ARG_S17R:	/* simm17,reg (loadaddri) */
			/* XXX: output $addr-.-4 ? */
			i += decode_imm(buf + i, len - i, op >> 6, -17);
			buf[i++] = ',';
			buf[i++] = ' ';
			i += decode_reg(buf + i, len - i, op >> 0);
			break;
		case ARG_U64R:	/* uimm64,reg (generic loadcons) */
		case ARG_S64R:	/* simm64,reg (generic loadconsx) */
			assert(0);
			break;
		case ARG_ORR:	/* [reg,]reg,reg (popcount, load[f], store[f], cshift) */
			if ((op >> 12) & 0x3f) {
				i += decode_reg(buf + i, len - i, op >> 12);
				buf[i++] = ',';
				buf[i++] = ' ';
			}
			i += decode_reg(buf + i, len - i, op >> 6);
			buf[i++] = ',';
			buf[i++] = ' ';
			i += decode_reg(buf + i, len - i, op >> 0);
			break;
		case ARG_RRO:	/* reg,reg[,reg] (jmp{cc}) */
			i += decode_reg(buf + i, len - i, op >> 12);
			buf[i++] = ',';
			buf[i++] = ' ';
			i += decode_reg(buf + i, len - i, op >> 6);
			if (op & 0x3f) {
				buf[i++] = ',';
				buf[i++] = ' ';
				i += decode_reg(buf + i, len - i, op >> 0);
			}
			break;
		case ARG_RRR:	/* reg,reg,reg (common) */
			i += decode_reg(buf + i, len - i, op >> 12);
			buf[i++] = ',';
			buf[i++] = ' ';
			i += decode_reg(buf + i, len - i, op >> 6);
			buf[i++] = ',';
			buf[i++] = ' ';
			i += decode_reg(buf + i, len - i, op >> 0);
			break;
		case ARG_U8RR:	/* uimm8,reg,reg (common) */
			i += decode_imm(buf + i, len - i, op >> 12, 8);
			buf[i++] = ',';
			buf[i++] = ' ';
			i += decode_reg(buf + i, len - i, op >> 6);
			buf[i++] = ',';
			buf[i++] = ' ';
			i += decode_reg(buf + i, len - i, op >> 0);
			break;
		case ARG_S8RR:	/* simm8,reg,reg (muli, divi, modi?) */
			i += decode_imm(buf + i, len - i, op >> 12, -8);
			buf[i++] = ',';
			buf[i++] = ' ';
			i += decode_reg(buf + i, len - i, op >> 6);
			buf[i++] = ',';
			buf[i++] = ' ';
			i += decode_reg(buf + i, len - i, op >> 0);
			break;
		case ARG_S9RR:	/* simm9,reg,reg (loadi[f], storei[f]) */
			i += decode_imm(buf + i, len - i, op >> 12, -9);
			buf[i++] = ',';
			buf[i++] = ' ';
			i += decode_reg(buf + i, len - i, op >> 6);
			buf[i++] = ',';
			buf[i++] = ' ';
			i += decode_reg(buf + i, len - i, op >> 0);
			break;
		default:
			assert(0);
    }
    buf[i] = '\0';
    return i;
}

static int
decode_enum(char *buf, size_t len, unsigned op, unsigned bits, ...) {
	const char *arg;
	unsigned mask;
	unsigned i;
	va_list ap;

	assert(buf);
	assert(bits);
	mask = (1u << bits) - 1u;
	op &= mask;
	va_start(ap, bits);
	for (i = 0; i <= mask; i++) {
		arg = va_arg(ap, const char*);
		assert(arg);
		if (op == i) {
			for (i = 0; arg[i] && i < len; i++) {
				buf[i] = arg[i];
			}
			assert(!arg[i]);
			va_end(ap);
			return i;
		}
	}
	assert(0);
	return 0;
}

static int
decode_spfx(char *buf, size_t len, unsigned op) {
	return decode_enum(buf, len, op >> 21, 1, "", "s");
}

static int
decode_cond(char *buf, size_t len, unsigned op) {
	size_t i = 0;

#if DECODE_LONG_NAMES
	buf[i++] = '.';
	i += decode_enum(buf + i, len - i, op >> 21, 1, "", "not");
	i += decode_enum(buf + i, len - i, op >> 19, 2,
		"zero", "nan", "msb", "lsb");
#else
	i += decode_enum(buf + i, len - i, op >> 21, 1, "", "n");
	i += decode_enum(buf + i, len - i, op >> 19, 2, "z", "n", "m", "l");
#endif
	return i;
}

static int
decode_isiz(char *buf, size_t len, unsigned op) {
#if DECODE_SIZE_NUMERIC
	return decode_enum(buf, len, op >> 22, 2, ".8", ".16", ".32", ".64");
#else
	return decode_enum(buf, len, op >> 22, 2, ".b", ".d", ".q", DECODE_DEFCS_INT);
#endif
}

static int
decode_fsiz(char *buf, size_t len, unsigned op) {
#if DECODE_SIZE_NUMERIC
	return decode_enum(buf, len, op >> 22, 2, ".32", ".64", ".2?", ".3?" );
#else
	return decode_enum(buf, len, op >> 22, 2, ".f", DECODE_DEFCS_FP, ".2?", ".3?" );
#endif
}

static int
decode_fpop(const table_t *tp, char *buf, size_t len, unsigned op) {
	size_t i = 0;

	if (tp->simd) {
		i += decode_spfx(buf + i, len - i, op);
	}
	strcpy(buf + i, tp->name);
	i += strlen(tp->name);
	i += decode_enum(buf + i, len - i, op >> 20, 1, "", "x");
	i += decode_fsiz(buf + i, len - i, op);
	while (i < MAX_INSN_LEN) buf[i++] = ' ';
	return i + decode_args(buf + i, len - i, tp->args, op);
}

static int
decode_jump(const table_t *tp, char *buf, size_t len, unsigned op) {
	size_t i = 0;

	strcpy(buf + i, "jmp");
	i += 3;
	if (!(op & ((7 << 19) | 0770000))) {
		/* unconditional form */
		while (i < MAX_INSN_LEN) buf[i++] = ' ';
		return i + decode_args(buf + i, len - i, ARG_RO, op);
	}
	/* conditional form */
	i += decode_cond(buf + i, len - i, op);
	while (i < MAX_INSN_LEN) buf[i++] = ' ';
	return i + decode_args(buf + i, len - i, tp->args, op);
}

static int
decode_load(const table_t *tp, char *buf, size_t len, unsigned op) {
	size_t i = 0;

	strcpy(buf + i, tp->name);
	i += strlen(tp->name);
	i += decode_enum(buf + i, len - i, op >> 21, 1, "", "e");
	if (tp->args != ARG_S9RR && ((op >> 18) & 7)) {
		/* stream hints */
		buf[i++] = ((op >> 18) & 7) + '0';
	}
	i += decode_isiz(buf + i, len - i, op);
	while (i < MAX_INSN_LEN) buf[i++] = ' ';
	return i + decode_args(buf + i, len - i, tp->args, op);
}

static int
decode_clod(const table_t *tp, char *buf, size_t len, unsigned op) {
	size_t i = 0;

	strcpy(buf + i, tp->name);
	i += strlen(tp->name);
	i += decode_cond(buf + i, len - i, op);
	i += decode_isiz(buf + i, len - i, op);
	while (i < MAX_INSN_LEN) buf[i++] = ' ';
	return i + decode_args(buf + i, len - i, tp->args, op);
}

static int
decode_cons(const table_t *tp, char *buf, size_t len, unsigned op) {
	size_t i = 0;

	strcpy(buf + i, tp->name);
	i += strlen(tp->name);
	sprintf(buf + i, ".%u", (op >> 22) & 0x3);
	i += strlen(buf + i);
	while (i < MAX_INSN_LEN) buf[i++] = ' ';
	return i + decode_args(buf + i, len - i, tp->args, op);
}

static int
decode_nosz(const table_t *tp, char *buf, size_t len, unsigned op) {
	unsigned args = tp->args;
	size_t i = 0;

	strcpy(buf + i, tp->name);
	i += strlen(tp->name);
	switch (op >> 24) {
		case OP_SYSCALL:
			if (op & SYSCALL_TRAP) {
				strcpy(buf, "trap");
				i = strlen(buf);
			}
			break;
		case OP_LOADADDR:
			if (!(op & (LOADADDR_DATA | 07700))) {
				strcpy(buf, "loopentry");
				i = strlen(buf);
				args = ARG_R;
				break;
			}
			/* flow through */
		case OP_LOADADDRI:
			i += decode_enum(buf + i, len - i, op >> 23, 1, "", "d");
			break;
		case OP_SERIALIZE:
			i += decode_enum(buf + i, len - i, op >> 21, 1, "", "s");
			i += decode_enum(buf + i, len - i, op >> 22, 1, "", "x");
			i += decode_enum(buf + i, len - i, op >> 23, 1, "", "m");
			break;
	}
	while (i < MAX_INSN_LEN) buf[i++] = ' ';
	return i + decode_args(buf + i, len - i, args, op);
}

static int
decode_rop2(const table_t *tp, char *buf, size_t len, unsigned op) {
	const char *name = tp->name;
	size_t i = 0;

	if (tp->simd) {
		i += decode_spfx(buf + i, len - i, op);
	}
	if ((op >> 24) - OP_ROP2 < 8) {
		if ((op & ROP2_MODE_MASK) == ROP2_MODE_MUX) {
			if ((op >> 24) == OP_ROP2) {
				name = "mux";
			}
			else {
				name = "???";
			}
		}
	}
	strcpy(buf + i, name);
	i += strlen(name);
	if ((op >> 24) - OP_ROP2 < 8) {
#if DECODE_LONG_NAMES
		i += decode_enum(buf + i, len - i, op >> 18, 2, "", ".and", ".or", "");
#else
		i += decode_enum(buf + i, len - i, op >> 18, 2, "", "a", "o", "");
#endif
	}
	i += decode_isiz(buf + i, len - i, op);
	while (i < MAX_INSN_LEN) buf[i++] = ' ';
	return i + decode_args(buf + i, len - i, tp->args, op);
}

static int
decode_scan(const table_t *tp, char *buf, size_t len, unsigned op) {
	size_t i = 0;

	if (tp->simd) {
		i += decode_spfx(buf + i, len - i, op);
	}
#if DECODE_SCAN_CANONICAL
	strcpy(buf + i, tp->name);
	i += strlen(tp->name);
	i += decode_enum(buf + i, len - i, op >> 19, 1, "", "n");
	i += decode_enum(buf + i, len - i, op >> 18, 1, "", "r");
#else
	i += decode_enum(buf + i, len - i, op >> 18, 1, "lsb", "msb");
	i += decode_enum(buf + i, len - i, op >> 19, 1, "1", "0");
#endif
	i += decode_isiz(buf + i, len - i, op);
	while (i < MAX_INSN_LEN) buf[i++] = ' ';
	return i + decode_args(buf + i, len - i, tp->args, op);
}

static int
decode_simd(const table_t *tp, char *buf, size_t len, unsigned op) {
	unsigned args = tp->args;
	size_t i = 0;

	if (tp->simd) {
		i += decode_spfx(buf + i, len - i, op);
	}
	strcpy(buf + i, tp->name);
	i += strlen(tp->name);
	switch (op >> 24) {
		case OP_ADD:
			i += decode_enum(buf + i, len - i, op >> 18, 2, "", "c", "s", "?");
			break;
		case OP_SUB:
			i += decode_enum(buf + i, len - i, op >> 18, 2, "", "b", "f", "?");
			break;
		case OP_MUL:
		case OP_AMAC:
			i += decode_enum(buf + i, len - i, op >> 18, 1, "", "h");
			i += decode_enum(buf + i, len - i, op >> 19, 1, "", "s");
			break;
		case OP_DIV:
			i += decode_enum(buf + i, len - i, op >> 20, 1, "", "rem");
			i += decode_enum(buf + i, len - i, op >> 19, 1, "", "s");
			break;
		case OP_DIVI:
			i += decode_enum(buf + i, len - i, op >> 20, 1, "", "rem");
			buf[i++] = 'i';
			break;
		case OP_REM:
			i += decode_enum(buf + i, len - i, op >> 19, 1, "", "s");
			break;
		case OP_MAC:
			i += decode_enum(buf + i, len - i, op >> 18, 1, "l", "h");
			i += decode_enum(buf + i, len - i, op >> 19, 1, "", "s");
			break;
		case OP_CMPG:
		case OP_CMPLE:
		case OP_MAX:
		case OP_MIN:
		case OP_MINMAX:
			i += decode_enum(buf + i, len - i, op >> 20, 1, "", "s");
			break;
		case OP_CMPGI:
		case OP_CMPLEI:
		case OP_MAXI:
		case OP_MINI:
		case OP_MINMAXI:
			if (op & CMP_SIGNED) args = ARG_S8RR;
			i += decode_enum(buf + i, len - i, op >> 20, 1, "", "s");
			buf[i++] = 'i';
			break;
		case OP_INT2L:
		case OP_L2INT:
			i += decode_enum(buf + i, len - i, op >> 18, 2, "", "t", "f", "c");
			break;
		case OP_SHIFTL:
		case OP_SHIFTR:
		case OP_SHIFTRA:
		case OP_BITREV:
		case OP_ROTL:
		case OP_ROTR:
		case OP_DSHIFTL:
		case OP_DSHIFTR:
		case OP_DSHIFTRA:
		case OP_DBITREV:
			i += decode_enum(buf + i, len - i, op >> 20, 1, "", "h");
			break;
		case OP_MIX:
		case OP_EXPAND:
			i += decode_enum(buf + i, len - i, op >> 18, 2, "", "l", "h", "?");
			break;
		case OP_CSHIFT:
			i += decode_enum(buf + i, len - i, op >> 19, 1, "l", "r");
			break;
		case OP_F2INT:
		case OP_INT2F:
		case OP_D2INT:
		case OP_INT2D:
			i += decode_enum(buf + i, len - i, op >> 20, 1, "", "x");
			i += decode_enum(buf + i, len - i, op >> 18, 2, "", "t", "f", "c");
			break;
		case OP_CACHEMM:
			i += decode_enum(buf + i, len - i, op >> 21, 1, "f", "p");
			i += decode_enum(buf + i, len - i, op >> 20, 1, "", "l");
			i += decode_enum(buf + i, len - i, op >> 19, 1, "", "c");
			break;
		case OP_MOVE:
			if (!(op & ((7 << 19) | 0770000))) {
				/* unconditional form */
				args = ARG_RR;
				break;
			}
			/* conditional form */
			i += decode_cond(buf + i, len - i, op);
			break;
		case OP_VSEL:
			i += decode_enum(buf + i, len - i, op >> 20, 1, "", "h");
			break;
	}
	i += decode_isiz(buf + i, len - i, op);
	while (i < MAX_INSN_LEN) buf[i++] = ' ';
	return i + decode_args(buf + i, len - i, args, op);
}

static int
decode_triv(const table_t *tp, char *buf, size_t len, unsigned op) {
	size_t i = 0;

	strcpy(buf + i, tp->name);
	i += strlen(tp->name);
	while (i < MAX_INSN_LEN) buf[i++] = ' ';
	return i + decode_args(buf + i, len - i, tp->args, op);
}

static table_t
decode_table[256] = {
	/* arithmetic */
	[OP_ADD]         = { "add",			decode_simd, ARG_RRR,  1 },
	[OP_SUB]         = { "sub",			decode_simd, ARG_RRR,  1 },
	[OP_MUL]         = { "mul",			decode_simd, ARG_RRR,  1 },
	[OP_DIV]         = { "div",			decode_simd, ARG_RRR,  1 },
	[OP_ADDI]        = { "addi",		decode_simd, ARG_U8RR, 1 },
	[OP_SUBI]        = { "subi",		decode_simd, ARG_U8RR, 1 },
	[OP_MULI]        = { "muli",		decode_simd, ARG_S8RR, 1 },
	[OP_DIVI]        = { "div",			decode_simd, ARG_S8RR, 1 },
	[OP_REM]         = { "rem",			decode_simd, ARG_RRR,  1 },
	[OP_REMI]        = { "remi",		decode_simd, ARG_S8RR, 1 },
	[OP_MAC]         = { "mac",			decode_simd, ARG_RRR,  1 },
	[OP_AMAC]        = { "amac",		decode_simd, ARG_RRR,  1 },
	[OP_ADDSUB]      = { "addsub",		decode_simd, ARG_RRR,  1 },
	/* popcount */
	[OP_POPC]        = { "popcount",	decode_simd, ARG_ORR,  1 },
	[OP_POPCI]       = { "popcounti",	decode_simd, ARG_U8RR, 1 },
	/* increment & compare */
	[OP_INC]         = { "inc",			decode_simd, ARG_RR,   1 },
	[OP_DEC]         = { "dec",			decode_simd, ARG_RR,   1 },
	[OP_NEG]         = { "neg",			decode_simd, ARG_RR,   1 },
	[OP_SCAN]        = { "scan",		decode_scan, ARG_RR,   1 },
	[OP_CMPG]        = { "cmpg",		decode_simd, ARG_RRR,  1 },
	[OP_CMPLE]       = { "cmple",		decode_simd, ARG_RRR,  1 },
	[OP_CMPGI]       = { "cmpg",		decode_simd, ARG_U8RR, 1 },
	[OP_CMPLEI]      = { "cmple",		decode_simd, ARG_U8RR, 1 },
	[OP_ABS]         = { "abs",			decode_simd, ARG_RR,   1 },
	[OP_NABS]        = { "nabs",		decode_simd, ARG_RR,   1 },
	[OP_MAX]         = { "max",			decode_simd, ARG_RRR,  1 },
	[OP_MIN]         = { "min",			decode_simd, ARG_RRR,  1 },
	[OP_MINMAX]      = { "minmax",		decode_simd, ARG_RRR,  1 },
	[OP_MAXI]        = { "max",			decode_simd, ARG_U8RR, 1 },
	[OP_MINI]        = { "min",			decode_simd, ARG_U8RR, 1 },
	[OP_MINMAXI]     = { "minmax",		decode_simd, ARG_U8RR, 1 },
	/* LNS */
	[OP_LADD]        = { "ladd",		decode_simd, ARG_RRR,  1 },
	[OP_LSUB]        = { "lsub",		decode_simd, ARG_RRR,  1 },
	[OP_L2INT]       = { "l2int",		decode_simd, ARG_RR,   1 },
	[OP_INT2L]       = { "int2l",		decode_simd, ARG_RR,   1 },
	/* shift/rotate */
	[OP_SHIFTL]      = { "shiftl",		decode_simd, ARG_RRR,  1 },
	[OP_SHIFTR]      = { "shiftr",		decode_simd, ARG_RRR,  1 },
	[OP_SHIFTRA]     = { "shiftra",		decode_simd, ARG_RRR,  1 },
	[OP_SHIFTLI]     = { "shiftli",		decode_simd, ARG_U8RR, 1 },
	[OP_SHIFTRI]     = { "shiftri",		decode_simd, ARG_U8RR, 1 },
	[OP_SHIFTRAI]    = { "shiftrai",	decode_simd, ARG_U8RR, 1 },
	[OP_ROTL]        = { "rotl",		decode_simd, ARG_RRR,  1 },
	[OP_ROTR]        = { "rotr",		decode_simd, ARG_RRR,  1 },
	[OP_ROTLI]       = { "rotli",		decode_simd, ARG_U8RR, 1 },
	[OP_ROTRI]       = { "rotri",		decode_simd, ARG_U8RR, 1 },
	[OP_BITREV]      = { "bitrev",		decode_simd, ARG_RRR,  1 },
	[OP_BITREVI]     = { "bitrevi",		decode_simd, ARG_U8RR, 1 },
	[OP_DSHIFTL]     = { "dshiftl",		decode_simd, ARG_RRR,  1 },
	[OP_DSHIFTR]     = { "dshiftr",		decode_simd, ARG_RRR,  1 },
	[OP_DSHIFTRA]    = { "dshiftra",	decode_simd, ARG_RRR,  1 },
	[OP_DSHIFTLI]    = { "dshiftli",	decode_simd, ARG_U8RR, 1 },
	[OP_DSHIFTRI]    = { "dshiftri",	decode_simd, ARG_U8RR, 1 },
	[OP_DSHIFTRAI]   = { "dshiftrai",	decode_simd, ARG_U8RR, 1 },
	[OP_DBITREV]     = { "dbitrev",		decode_simd, ARG_RRR,  1 },
	[OP_DBITREVI]    = { "dbitrevi",	decode_simd, ARG_U8RR, 1 },
	/* bitop */
	[OP_BTST]        = { "btst",		decode_simd, ARG_RRR,  1 },
	[OP_BCLR]        = { "bclr",		decode_simd, ARG_RRR,  1 },
	[OP_BCHG]        = { "bchg",		decode_simd, ARG_RRR,  1 },
	[OP_BSET]        = { "bset",		decode_simd, ARG_RRR,  1 },
	[OP_BTSTI]       = { "btsti",		decode_simd, ARG_U8RR, 1 },
	[OP_BCLRI]       = { "bclri",		decode_simd, ARG_U8RR, 1 },
	[OP_BCHGI]       = { "bchgi",		decode_simd, ARG_U8RR, 1 },
	[OP_BSETI]       = { "bseti",		decode_simd, ARG_U8RR, 1 },
	/* bytewise shift */
	[OP_MIX]         = { "mix",			decode_simd, ARG_RRR,  0 },
	[OP_EXPAND]      = { "expand",		decode_simd, ARG_RRR,  0 },
	[OP_CSHIFT]      = { "cshift",		decode_simd, ARG_ORR,  0 },
	[OP_BYTEREV]     = { "byterev",		decode_simd, ARG_RR,   1 },
	[OP_VSEL]        = { "vsel",		decode_simd, ARG_RRR,  1 },
	[OP_VSELI]       = { "vseli",		decode_simd, ARG_U8RR, 1 },
	/* ROP2 */
	[OP_AND]         = { "and",			decode_rop2, ARG_RRR,  1 },
	[OP_ANDN]        = { "andn",		decode_rop2, ARG_RRR,  1 },
	[OP_NAND]        = { "nand",		decode_rop2, ARG_RRR,  1 },
	[OP_NOR]         = { "nor",			decode_rop2, ARG_RRR,  1 },
	[OP_OR]          = { "or",			decode_rop2, ARG_RRR,  1 },
	[OP_ORN]         = { "orn",			decode_rop2, ARG_RRR,  1 },
	[OP_XNOR]        = { "xnor",		decode_rop2, ARG_RRR,  1 },
	[OP_XOR]         = { "xor",			decode_rop2, ARG_RRR,  1 },
	/* ROP2I */
	[OP_ANDI]        = { "andi",		decode_rop2, ARG_S9RR, 1 },
	[OP_ANDNI]       = { "andni",		decode_rop2, ARG_S9RR, 1 },
	[OP_NANDI]       = { "nandi",		decode_rop2, ARG_S9RR, 1 },
	[OP_NORI]        = { "nori",		decode_rop2, ARG_S9RR, 1 },
	[OP_ORI]         = { "ori",			decode_rop2, ARG_S9RR, 1 },
	[OP_ORNI]        = { "orni",		decode_rop2, ARG_S9RR, 1 },
	[OP_XNORI]       = { "xnori",		decode_rop2, ARG_S9RR, 1 },
	[OP_XORI]        = { "xori",		decode_rop2, ARG_S9RR, 1 },
	/* FPU level 1 */
	[OP_FADD]        = { "fadd",		decode_fpop, ARG_RRR,  1 },
	[OP_FSUB]        = { "fsub",		decode_fpop, ARG_RRR,  1 },
	[OP_FMUL]        = { "fmul",		decode_fpop, ARG_RRR,  1 },
	[OP_F2INT]       = { "f2int",		decode_simd, ARG_RR,   0 },
	[OP_INT2F]       = { "int2f",		decode_simd, ARG_RR,   0 },
	[OP_D2INT]       = { "d2int",		decode_simd, ARG_RR,   0 },
	[OP_INT2D]       = { "int2d",		decode_simd, ARG_RR,   0 },
	[OP_FIAPRX]      = { "fiaprx",		decode_fpop, ARG_RR,   1 },
	[OP_FSQRTIAPRX]  = { "fsqrtiaprx",	decode_fpop, ARG_RR,   1 },
	/* FPU level 2 */
	[OP_FDIV]        = { "fdiv",		decode_fpop, ARG_RRR,  1 },
	[OP_FSQRT]       = { "fsqrt",		decode_fpop, ARG_RR,   1 },
	/* FPU level 3 */
	[OP_FLOG]        = { "flog",		decode_fpop, ARG_ORR,  1 },
	[OP_FEXP]        = { "fexp",		decode_fpop, ARG_ORR,  1 },
	[OP_FMAC]        = { "fmac",		decode_fpop, ARG_RRR,  1 },
	[OP_FADDSUB]     = { "faddsub",		decode_fpop, ARG_RRR,  1 },
	/* load/store */
	[OP_LOAD]        = { "load",		decode_load, ARG_ORR,  0 },
	[OP_STORE]       = { "store",		decode_load, ARG_ORR,  0 },
	[OP_LOADF]       = { "loadf",		decode_load, ARG_ORR,  0 },
	[OP_STOREF]      = { "storef",		decode_load, ARG_ORR,  0 },
	[OP_LOADI]       = { "loadi",		decode_load, ARG_S9RR, 0 },
	[OP_LOADIF]      = { "loadif",		decode_load, ARG_S9RR, 0 },
	[OP_STOREI]      = { "storei",		decode_load, ARG_S9RR, 0 },
	[OP_STOREIF]     = { "storeif",		decode_load, ARG_S9RR, 0 },
	/* XXX: cload/cstore missing */
	[OP_CLOAD]		 = { "cload",		decode_clod, ARG_RRR,  0 },
	[OP_CSTORE]		 = { "cstore",		decode_clod, ARG_RRR,  0 },
	/* XXX: deprecated */
	[OP_CACHEMM]     = { "cachemm",		decode_simd, ARG_RR,   0 },
	/* move */
	[OP_MOVE]        = { "move",		decode_simd, ARG_RRR,  0 },
	[OP_WIDEN]       = { "widen",		decode_simd, ARG_RR,   0 },
	/* loadcons */
	[OP_LOADCONS]    = { "loadcons",	decode_cons, ARG_U16R, 0 },
	[OP_LOADCONSX]   = { "loadconsx",	decode_cons, ARG_S16R, 0 },
	/* loadaddr */
	[OP_LOADADDR]    = { "loadaddr",	decode_nosz, ARG_RR,   0 },
	[OP_LOADADDRI]   = { "loadaddri",	decode_nosz, ARG_S17R, 0 },
	/* get/put */
	[OP_GET]         = { "get",			decode_triv, ARG_RR,   0 },
	[OP_PUT]         = { "put",			decode_triv, ARG_RR,   0 },
	[OP_GETI]        = { "geti",		decode_triv, ARG_U16R, 0 },
	[OP_PUTI]        = { "puti",		decode_triv, ARG_U16R, 0 },
	/* loadm/storem */
	[OP_LOADM]       = { "loadm",		decode_triv, ARG_RRR,  0 },
	[OP_STOREM]      = { "storem",		decode_triv, ARG_RRR,  0 },
	/* jmp */
	[OP_JMP]         = { "jmp",			decode_jump, ARG_RRO,  0 },
	[OP_LOOP]        = { "loop",		decode_triv, ARG_RR,   0 },
	[OP_SYSCALL]     = { "syscall",		decode_nosz, ARG_U16R, 0 },
	[OP_HALT]        = { "halt",		decode_triv, ARG_NONE, 0 },
	[OP_RFE]         = { "rfe",			decode_triv, ARG_NONE, 0 },
	/* control */
	[OP_SRB_SAVE]    = { "srb_save",	decode_triv, ARG_NONE, 0 },
	[OP_SRB_RESTORE] = { "srb_restore",	decode_triv, ARG_NONE, 0 },
	[OP_SERIALIZE]   = { "serialize",	decode_nosz, ARG_NONE, 0 },
	/* nop */
	[OP_NOP]         = { "nop",			decode_triv, ARG_NONE, 0 },
};

int
fcpu_decode_instruction(char *buf, size_t len, unsigned op) {
	char mybuffer[512];	/* more than enough */
	const table_t *tp;
	size_t n;
	fcpu_used_register[0] = -1;
	fcpu_used_register[1] = -1;
	fcpu_used_register[2] = -1;
	fcpu_used_register_nb = 0;
	
	tp = &decode_table[(op >> 24) & 0xff];
	if (tp->name) {
		assert(tp->decode);
		n = tp->decode(tp, mybuffer, sizeof(mybuffer), op);
	}
	else {
		strcpy(mybuffer, "???");
		n = 3;
	}
	if (n >= len) {
		return -1;
	}
	memcpy(buf, mybuffer, n);
	buf[n] = '\0';
	return n;
}

const char*
fcpu_opcode_name(unsigned op) {
	return decode_table[(op >> 24) & 0xff].name;
}
