/*
 * emumain.c -- F-CPU instruction-level emulator main program
 * Copyright (C) 2002, 2003 Michael Riepe <michael@stud.uni-hannover.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

static const char rcsid[] = "@(#) $Id: emumain.c,v 1.5 2003/02/01 14:01:36 michael Exp $";

#if HAVE_CONFIG_H
#include <config.h>
#endif

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
void *malloc();
unsigned long strtoul();
#endif

#include <stdio.h>

#if HAVE_UNISTD_H
#include <unistd.h>
#else
int read(), write();
int getopt(), optind;
char *optarg;
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#else
extern int errno;
#endif

#if HAVE_MATH_H
#include <math.h>
#else
int isnan();
double sqrt(), log(), exp();
#endif

#if HAVE_FCNTL_H
#include <fcntl.h>
#else
int open(), close();
#endif
#ifndef O_RDONLY
#define O_RDONLY 0
#endif

#if HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#if HAVE_SYS_MMAN_H
#include <sys/mman.h>
#ifndef MAP_FAILED
#define MAP_FAILED (-1)
#endif
#if !defined(MAP_ANONYMOUS) && defined(MAP_ANON)
#define MAP_ANONYMOUS	MAP_ANON
#endif
#endif

#include <fcpu_opcodes/fcpu_opcodes.h>
#include "emu.h"

#if HOST_BIG_ENDIAN
U64 ftohll(U64 x)
{
  U64 rv;
  U8 *s = (U8*)&x, *d =  (U8*)&rv;
  int i;
  for(i = 0; i < 7; i++)
    d[i] = s[7-i];

  return rv;
}
#else
#define ftohll(x) (x)
#endif



int L1Icount = 0;
int TotCyclesWaited = 0;
int cycles[1024];
char cycles_s[1024][1024];


void wait1cycle()
{
  pf_doPrefetch();
  lsu_doPrefetch();//TODO a mettre dans le core
  wait(); // pour que systemc augmente le temps
}

U32 realFetch()
{
  U32 rv = pf_getNextInstruction();
  int cycleWaited = 0;
  U64 pc = regs.r_pc.C(o,0);
  pf_doPrefetch();
  while(waitPrefetcher)
    {
      cycleWaited++;
      rv = pf_getNextInstruction();
      wait1cycle() ;
    }
  if(cycles[pc])
    {
      char buf[1024];
      strcpy(buf,cycles_s[pc]);
      sprintf(cycles_s[pc],"%s+%d",buf,cycleWaited+1);
    }
  else
    sprintf(cycles_s[pc],"%d",cycleWaited+1);
  cycles[regs.r_pc.C(o,0)] += cycleWaited+1;
  TotCyclesWaited += cycleWaited;
  return rv;
}

unsigned char lsu_read(U64 addr, int num_reg) {
  
  U64 pc = regs.r_pc.C(o,0);
  unsigned char data = lsu_do_read(addr,num_reg);
  int cycleWaited = 0;
  while(waitDataCache)
    {
      cycleWaited++;
      data = lsu_do_read(addr,num_reg);
      wait1cycle();
    }
  /* here we don't add a 1-cycle wait because it has been counted during the instruction fetch */
  if (cycleWaited) 
    {
      char escape1[] = { '(',0,};//0x1B,0x5B,'3','3','m',0};
      char escape2[] = { ')',0};//0x1B,0x5B,'0','m',0};
      if(cycles[pc])
	{
	  char buf[1024];
	  strcpy(buf,cycles_s[pc]);
	  sprintf(cycles_s[pc],"%s+%s%d%s",buf,escape1,cycleWaited,escape2); /* We add stars in the display coz Data (and not Instr) */
	}
      else
	sprintf(cycles_s[pc],"%s%d%s",escape1,cycleWaited,escape2);
      cycles[regs.r_pc.C(o,0)] += cycleWaited ; 
      TotCyclesWaited += cycleWaited;
    }
  return data ;
}

void lsu_write(U64 addr, int num_reg, unsigned char data) {
  U64 pc = regs.r_pc.C(o,0);
  int cycleWaited = 0;
  lsu_do_write(addr,num_reg,data);
  while(waitDataCache)
    {
      cycleWaited++;
      lsu_do_write(addr,num_reg,data);
      wait1cycle();
    }
 /* here we don't add a 1-cycle wait because it has been counted during the instruction fetch */
  if (cycleWaited) 
    {
      char escape1[] = { '(',0,};//0x1B,0x5B,'3','3','m',0};
      char escape2[] = { ')',0};//0x1B,0x5B,'0','m',0};
      if(cycles[pc])
	{
	  char buf[1024];
	  strcpy(buf,cycles_s[pc]);
	  sprintf(cycles_s[pc],"%s+%s%d%s",buf,escape1,cycleWaited,escape2); /* We add stars in the display coz Data (and not Instr) */
	}
      else
	sprintf(cycles_s[pc],"%s%d%s",escape1,cycleWaited,escape2);
      cycles[regs.r_pc.C(o,0)] += cycleWaited ; 
      TotCyclesWaited += cycleWaited;
    }
}

void
show(unsigned long ip, unsigned insn) {
	char buf[1024];
	int i;
	if (fcpu_decode_instruction(buf, sizeof(buf), insn) < 0) {
		abort();
	}
	fprintf(stderr, "%08lx  %02x%02x%02x%02x  %s", ip, (U8)(insn>>24),
			(U8)(insn>>16), (U8)(insn>>8), (U8)(insn>>0), buf);
	if(fcpu_used_register_nb)
	  {
	    fprintf(stderr,"--");
	    for(i = 0; i < fcpu_used_register_nb; i++)
	      fprintf(stderr,"r%d = %llX ",fcpu_used_register[i],r(fcpu_used_register[i]).C(o,0));
	  }
	fprintf(stderr,"\n");
}
void dumpprofile()
{
  char buf[1024];
  unsigned long ip,  insn;
  ip = 0;
  do
    {
      insn = fetch(ip);
      if (fcpu_decode_instruction(buf, sizeof(buf), insn) < 0) {
	abort();
      }
      fprintf(stderr, "%02d    =             %s\n    %04lx   %s\n",cycles[ip],cycles_s[ip], ip, buf);
      ip+=4;
      if(ip % (4*4) ==0)
	fprintf(stderr, "--\n");
	
    }
  while(insn != 0x40000000);
  fprintf(stderr, "%d\n",TotCyclesWaited);
}
void
exception(unsigned long ip, unsigned insn, unsigned code) {
	static const char *msgs[EX_number] = {
		[EX_ACCESS]    = "access rights violation",
		[EX_ADDRESS]   = "address out of range",
		[EX_ALIGNMENT] = "misaligned address",
		[EX_INVALID]   = "invalid instruction",
		[EX_ZERO]      = "divide by 0",
		[EX_RANGE]     = "value out of range",
		[EX_HALT]      = "halted",
		[EX_NOHAND]    = "no exception handler",
	};

	if (code >= EX_number || !msgs[code]) {
		fprintf(stderr, "\n*** exception %u ***\n", code);
	}
	else {
	  fprintf(stderr, "\n*** %s ***\n", msgs[code]);
	  printf("%d\n",TotCyclesWaited);
	  lsu_flush();
	  dumpprofile();
	}
}

#define fcpu_io(rw, func, a0, a1, a2, a3)	\
	do { \
		int fd = a1; \
		size_t n = a3; \
		void *p = memmap(a2, 0, a3, rw); \
		if (fd != a1 || n != a3 || !p || excode != EX_NONE) { \
			a0 = -EINVAL; \
			ex(EX_NONE); \
			break; \
		} \
		n = func(fd, p, n); \
		if (n == (size_t)-1) { \
			a0 = -errno; \
			break; \
		} \
		a0 = n; \
	} while (0)

/* provide simple I/O via syscall */
int syscall_handler(U32 opcode) {
	if (UIMM16 != 0) {
		/* let emulator handle other syscalls */
		return 0;
	}
	if (r(1).C(q,0) != r(1).C(o,0)) {
		/* syscall number too big */
		r(1).C(o,0) = -ENOSYS;
		return 1;
	}
	switch (r(1).C(q,0)) {
		case 0:
			fprintf(stderr, "program exited with status %u\n", r(2).C(b,0));
			exit(0);
		case 1:
			fcpu_io(1, read, r(1).C(o,0), r(2).C(o,0), r(3).C(o,0), r(4).C(o,0));
			break;
		case 2:
			fcpu_io(0, write, r(1).C(o,0), r(2).C(o,0), r(3).C(o,0), r(4).C(o,0));
			break;
		default:
			r(1).C(o,0) = -ENOSYS;
			break;
	}
	return 1;
}

static const char usage[] = "usage: emu [-i] [-m mbytes] binfile\n";


int interactive = 0;
void
fcpu_init() 
{

  lsu_init() ;

    fputs("F-CPU instruction-level emulator " VERSION_STRING "\n"
	  "Copyright (C) 2002, 2003 Michael \"Tired\" Riepe\n"
	  "This program is free software; you can redistribute it and/or\n"
	  "modify it under the terms of the GNU General Public License.\n"
	  "This program is distributed WITHOUT ANY WARRANTY.\n\n",
	  stderr);
    fprintf(stderr, "Emulated F-CPU has %u-bit registers\n", 8 * MAXSIZE);
    fprintf(stderr, "and uses %s write semantics.\n\n", PARTIAL_WRITES ? "old" : "new");
    fprintf(stderr, "Enter \"?\" for help.\n\n");
    initemu();
    syscall_hook = syscall_handler;
  }
void
fcpu_main() 
  {
    int trace = 0;
    U64 ip;
    U32 insn;
    for (;;) {
      int done = 0;
      unsigned i;

      ex(EX_NONE);
      ip = regs.r_pc.C(o,0);
      insn = realFetch(ip);
      if (!interactive) {
	while (!excode && !interactive) {
	  if (trace) {
	    fprintf(stderr,"%d\n",TotCyclesWaited);
	    pf_showPrefetchLines();
	    show(ip, insn);
	  }
	  regs.r_pc.C(o,0) = ip + 4;
	  emulate1(insn);
	  if(trace && fcpu_used_register_nb)
	    {
	      fprintf(stderr,"-->");
	      for(i = 0; i < fcpu_used_register_nb; i++)
		fprintf(stderr,"r%d = %llX ",fcpu_used_register[i],r(fcpu_used_register[i]).C(o,0));
	      fprintf(stderr,"\n");
	    }

	  TotCyclesWaited ++;
	  if (excode) break;
	  ip = regs.r_pc.C(o,0);
	  insn = realFetch();//fetch(ip);
	  //if( ip == 0xd8)
	  // interactive = true;
	}
	regs.r_pc.C(o,0) = ip;
	interactive = 1;
	trace = 0;
      }
      if (excode) {
	exception(ip, insn, excode);
	ex(EX_NONE);
      }
      show(ip, insn);
      do {
	char line[1024], *s;
	int regno;
	   
	fprintf(stderr, ">");
	if (!fgets(line, sizeof(line), stdin)) {
	  exit(0);
	}
	switch (*line) {
	case '\n':
	  continue;
	case 'g':
	  if (line[1] == '=') {
	    regs.r_pc.C(o,0) = strtoul(line + 2, NULL, 16);
	  }
	  interactive = 0;
	  done = 1;
	  break;
	case 'l':
	  if (line[1] != '\n') {
	    ip = strtoul(line + 1, NULL, 16);
	    insn = fetch(ip);// this is not for beeing executed..
	  }
	  for (i = 0; i < 16; i++) {
	    show(ip, insn);
	    insn = fetch(ip += 4);// this is not for beeing executed..
	  }
	  ex(EX_NONE);
	  break;
	case 'q':
	  exit(0);
	case 'r':
	  regno = strtoul(line + 1, &s, 10);
	  if (s == line + 1 || regno > 63u) break;
	  printf("r%d=%16llx\n", regno, r(regno).C(o,0));
	  break;

	case 's':
	  if (line[1] == '=') {
	    regs.r_pc.C(o,0) = strtoul(line + 2, NULL, 16);
	  }
	  ip = regs.r_pc.C(o,0);
	  insn = fetch(ip);
					
	  if (excode) {
	    exception(ip, insn, excode);
	    ex(EX_NONE);
	    show(ip, insn);
	    break;
	  }
	  regs.r_pc.C(o,0) = ip + 4;
	  emulate1(insn);
	  if (excode) {
	    exception(ip, insn, excode);
	    regs.r_pc.C(o,0) = ip;
	    ex(EX_NONE);
	    show(ip, insn);
	    break;
	  }
	  done = 1;
	  break;
	case 't':
	  if (line[1] == '=') {
	    {
	      regs.r_pc.C(o,0) = strtoul(line + 2, NULL, 16);
	    }
	  }else
	    regs.r_pc.C(o,0) +=4;
	  interactive = 0;
	  trace = 1;
	  done = 1;
	  break;
	case '?':
	  fprintf(stderr, "usage:\n"
		  "  ?          show this help\n"
		  "  g[=addr]   run (from addr)\n"
		  "  l[addr]    disassemble (from addr)\n"
		  "  q          quit\n"
		  "  r{n}       show contents of register {n}\n"
		  "  s[=addr]   single-step (from addr)\n"
		  "  t[=addr]   trace (from addr)\n"
		  "  dp         show the L0 instruction cache\n"
		  "  dd         dump the L/SU\n"
		  "  dm         dump memory\n"
		  );
	  break;
	case 'd':
          {
	    int ok = 1 ;
	  switch(line[1]) {
	  case 'p':
	    pf_showPrefetchLines();
	    break;
	  case 'm':
	    // dump the memory to verify the result of the program
	    {
	      int addr= 0;
	      for (addr =0x200; addr<0x300; addr+=8*4)
		{
		  U64 *p = (U64*)memmap(addr, 0, 30, 0);
		  
		  int i;
		  fprintf(stderr,"\n%X :",(int)addr);
		  for(i = 0; i <4; i++)
		    fprintf(stderr,"%lld ",ftohll(p[i]));
		}
	      fprintf(stderr,"\n");
	    }
	    break;
	  case 'd':
	    lsu_dump() ;
	    break;
	  default:
	    ok = 0 ;
	  }
	  if (ok) break ;
	  }
	default:
	  fprintf(stderr, "huh?\n");
	}
      }
      while (!done);
    }
  }
  
