/*
 * emu.h -- definitions for F-CPU instruction-level emulator core
 * Copyright (C) 2002, 2003 Michael Riepe <michael@stud.uni-hannover.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* @(#) $Id: emu.h,v 1.7 2003/01/29 19:31:06 michael Exp $ */

#ifndef _EMU_H
#define _EMU_H

#ifdef __cplusplus
extern "C" { /* I really dislike this - iwj. */
#endif

#if WORDS_BIGENDIAN
#define HOST_BIG_ENDIAN	1
#else /* WORDS_BIGENDIAN */
#define HOST_BIG_ENDIAN	0
#endif /* WORDS_BIGENDIAN */

#ifndef LOG_MAXSIZE
#define LOG_MAXSIZE			3	/* emulate a 64-bit F-CPU by default */
#endif /* LOG_MAXSIZE */

#define MAXSIZE				(1u << LOG_MAXSIZE)
#define PARTIAL_WRITES		0	/* new write semantics: high part is cleared */

#define NUM_REG 64

  typedef   signed char      I8;
  typedef unsigned char      U8;
  typedef          short     I16;
  typedef unsigned short     U16;
  typedef          int       I32;
  typedef unsigned int       U32;
#if SIZEOF_LONG == 8
  typedef          long      I64;
  typedef unsigned long      U64;
#else
  typedef          long long I64;
  typedef unsigned long long U64;
#endif

#define ST_b	I8
#define ST_d	I16
#define ST_q	I32
#define ST_o	I64
#define ST(sz)	ST_##sz

#define UT_b	U8
#define UT_d	U16
#define UT_q	U32
#define UT_o	U64
#define UT(sz)	UT_##sz

#define FT_F	float
#define FT_D	double
#define FT(sz)	FT_##sz

#define BYTES_b		(sizeof(I8))
#define BYTES_d		(sizeof(I16))
#define BYTES_q		(sizeof(I32))
#define BYTES_o		(sizeof(I64))
#define BYTES_F		(sizeof(float))
#define BYTES_D		(sizeof(double))
#define BYTES(sz)	BYTES_##sz

  /* number of chunks */
#define CHUNKS(sz)	(MAXSIZE/BYTES(sz))

#if SIZEOF_LONG == 8
  /* 64-bit host */
#define best		o
#define ST_best		ST_o
#define UT_best		UT_o
#define CHUNKS_best	CHUNKS_o
#define BYTES_best	BYTES_o
#else
  /* 32-bit host */
#define best		q
#define ST_best		ST_q
#define UT_best		UT_q
#define CHUNKS_best	CHUNKS_q
#define BYTES_best	BYTES_q
#endif

#undef BITS
#define BITS(sz)	(8*BYTES(sz))

#define none /**/

  /* opcode fields */
#define R1	((opcode >>	 0) & 077)	/* destination */
#define R2	((opcode >>	 6) & 077)	/* first operand */
#define R3	((opcode >> 12) & 077)	/* second operand */
#define RA	(R1 ^ 1)				/* alternate destination */
#define IL	(opcode >>	6)			/* long (16+ bit) immediate */
#define IS	(opcode >> 12)			/* short (8+ bit) immediate */

  /* immediate constants */
#define UIMM8	((U8)IS)
#define SIMM8	((I8)IS)
#define SIMM9	((I16)(IS << 7) >> 7)
#define UIMM16	((U16)IL)
#define SIMM16	((I16)IL)
#define SIMM17	((I32)(IL << 15) >> 15)

  /* a single register */
  union reg {
    U8		b[CHUNKS(b)];
    U16		d[CHUNKS(d)];
    U32		q[CHUNKS(q)];
    U64		o[CHUNKS(o)];
    I8		sb[CHUNKS(b)];
    I16		sd[CHUNKS(d)];
    I32		sq[CHUNKS(q)];
    I64		so[CHUNKS(o)];
    float	F[CHUNKS(F)];
    double	D[CHUNKS(D)];
  };

  /* the register set */
  extern struct regs {
    union reg	r[64];
    union reg	r_pc;
  } regs;

  /* shortcuts */
#define r(x)					regs.r[x]
#define for_all_chunks(i,sz)	for ((i) = 0; (i) < CHUNKS(sz); (i)++)

  /* chunk selector */
#if HOST_BIG_ENDIAN
#define C(sz,i)		sz[CHUNKS(sz)-1-(i)]
#define C2(sz,i)	sz[2*CHUNKS(sz)-1-(i)]
#define SC(sz,i)	s##sz[CHUNKS(sz)-1-(i)]
#else
#define C(sz,i)		sz[i]
#define C2(sz,i)	sz[i]
#define SC(sz,i)	s##sz[i]
#endif

  /* byte `j' of `sz'-sized chunk `i' of register `R' */
#define cbyte(R,sz,i,j)	(((U8*)&R.C(sz,i))[j])

  /* result initializer */
#if PARTIAL_WRITES
#define reginit(R)	r(R)	/* preload result with original contents */
#else
#define reginit(R)	{{0}}	/* preload result with zero */
#endif

  /* special registers */
  enum {
    SR_NUMBERS,
    SR_FAMILY,
    SR_STEPPING,
    SR_MAX_SIZE,
    SR_SIZE_0,
    SR_SIZE_1,
    SR_SIZE_2,
    SR_SIZE_3,
    SR_MAX_CHUNK_SIZE,
    SR_CYCLE,
    SR_PAGING,
    SR_CONTROL,
    SR_IRQ_BASE,
    SR_IRQ_SIZE,
    SR_TRAP_BASE,
    SR_TRAP_SIZE,
    SR_SYSCALL_BASE,
    SR_SYSCALL_SIZE,
    SR_TLBMISS_BASE,
    SR_URL,
    SR_URL_last = SR_URL + 7,
    SR_LAST_SR
  };

#define _SR_RD	(1u << 0)
#define _SR_WR	(1u << 1)
#define _SR_RW	(_SR_RD|_SR_WR)

  extern struct sreg {
    U32 p_super;
    U32 p_user;
    union {
      U64 v;		/* XXX: use `union reg' instead?! */
      char s[8];
    } u;
  } sregs[];

  /* exception codes */
  enum {
    EX_NONE,
    EX_ACCESS,
    EX_ADDRESS,
    EX_ALIGNMENT,
    EX_INVALID,
    EX_ZERO,
    EX_RANGE,
    EX_HALT,
    EX_NOHAND,
    EX_number
  };

  /* raise an exception */
#define ex(x)	(excode = (x))

  extern unsigned excode;

  extern int (*syscall_hook)(U32 opcode);
  extern int (*trap_hook)(U32 opcode);

  extern void initemu(void);
  extern int emulate1(U32 opcode);


  typedef int BOOL;

#include "lsu.h"
#include "prefetcher.h"

  // instruction cache L1
  // implemented to wait 5 cycles.
  extern BOOL waitDataCache ;
  extern BOOL waitPrefetcher ;

  U32 getL1Data(int i);
  BOOL L1DoLoadMem(U64 addr);
  U32 fetch(U64 ip) ; // get instruction directly from memory (no L0 cache)
  unsigned char data_load(U64 addr) ;
  void data_store(U64 addr, unsigned char data) ;

  BOOL L1DataGetData(U64 addr, void * target) ;
  BOOL L1DataDoWriteMem(U64 addr, void * source) ;
  
  void fcpu_main();
  void fcpu_init();
  extern void wait();
  void*
  memmap(U64 virtaddr, U64 align, U64 len, int write_mode) ;
#ifdef __cplusplus
} // extern "C"
#endif

#endif /* _EMU_H */
